(module demo-app/public-actions
    ()
  (import srfi-1)
  (import srfi-11)
  (import srfi-19)

  (import pattern-matching)

  (import lshift/publish-pattern)

  (import lshift/db)
  (import demo-app/models)

  (import lshift/xslt-ui)
  (import lshift/field-parameters)
  (import lshift/db-validated-fields)

  (import lshift/common)
  (import lshift/error)
  (import utils/sxml-tools)
  (import xxexpr)

  (define (date=? t1 t2)
    (let ((d1 (time-monotonic->date t1))
	  (d2 (time-monotonic->date t2)))
      (and (= (date-day d1) (date-day d2))
	   (= (date-month d1) (date-month d2))
	   (= (date-year d1) (date-year d2)))))

  (define (length>? l n)
    (cond
     ((zero? n) (pair? l))
     ((null? l) #f)
     (else (length>? (cdr l) (- n 1)))))

  (define (gather n l)
    (if (length>? l n)
	(let-values (((head tail) (split-at l n)))
	  (cons head (gather n tail)))
	(list l)))

  (define dummy-schema
    '(data
      (application_data ...)
      (registration ...)
      (socialfunctions (field ...))
      (rate-groups (group (registration_rate_group (name (field ...)))
			  (registration_rate (guid (field ...))
					     (name (field ...))
					     (price (field ...)))))
      (days (day (date (field ...))
		 (session-row (session-block (timeslot (starttime (field ...))
						       (endtime (field ...)))
					     (registration_session (registration_guid (field ...))
								   (session_guid (field ...)))
					     (session (guid (field ...))
						      (title (field ...)))
					     ...)
			      ...
			      (social-functions (rowspan 123)
						(socialfunction (guid (field ...))
								(starttime (field ...))
								(price (field ...))
								(title (field ...)))
						...))
		 ...)
	    ...)
      (accommodation (required (field ...))
		     (yes (field ...))
		     (no (field ...)))
      (dataprotection (dataprotection (field ...))
		      (yes (field ...))
		      (no (field ...)))
      (hotel-rows (row (hotel (name (field ...))
			      (walkingdistance (field ...)))
		       (hotelroom (hotel_guid (field ...))
				  (roomtype (field ...))
				  (price (field ...)))
		       ...))
      (dietaryrequirements (field ...))
      (diet (dietaryrequirement (guid (field ...))
				(name (field ...)))
	    ...)
      ))

  (define (make-session-block entry)
    (if (null? entry)
	`(session-block)
	(let ((timeslot (first entry))
	      (selection (second entry))
	      (sessions (third entry)))
	  `(session-block ,(ui-db-result/ro timeslot)
			  ,(ui-db-result selection
					 (build-all-parameters
					  `((session_guid required1))))
			  ,@(map ui-db-result/ro sessions)))))

  (define (make-session-rows date-for-socialfunctions timeslots-and-sessions)
    (let* ((timeslot-count (length timeslots-and-sessions))
	   (timeslots-per-row 3)
	   (empty-row (make-list timeslots-per-row '()))
	   (rows (gather timeslots-per-row timeslots-and-sessions))
	   (row-count (length rows))
	   (socialfunctions (get-socialfunctions-on-date date-for-socialfunctions)))
      (let ((session-rows (map (lambda (row)
				 `(session-row ,@(map make-session-block
						      (take (append row empty-row)
							    timeslots-per-row))))
			       rows)))
	(match session-rows
	  (((session-row ,block ...) ,rest ...)
	   `((session-row ,@block (social-functions (rowspan ,row-count)
						    ,@(map ui-db-result/ro socialfunctions)))
	     ,@rest))))))

  (define (get-session-blocks-for-ui)
    (let* ((all-timeslots-and-sessions (get-timeslots-and-sessions))
	   (timeslots-and-sessions-with-selection
	    (map (lambda (timeslot-and-sessions)
		   (list (car timeslot-and-sessions)
			 (make-db-result demo-app:registration_session)
			 (cadr timeslot-and-sessions)))
		 all-timeslots-and-sessions))
	   (grouped-timeslots (split-by-equivalence-class
			       (lambda (e1 e2)
				 (date=? (db-result-get (car e1) 'starttime)
					 (db-result-get (car e2) 'starttime)))
			       timeslots-and-sessions-with-selection)))
      (values
       `(days ,@(map (lambda (timeslots-and-sessions)
		       (let ((first-starttime (db-result-get (car (first timeslots-and-sessions))
							     'starttime)))
			 `(day ,(readonly-vf 'date (date->string (time-monotonic->date
								  first-starttime)
								 "~A ~e ~B ~Y"))
			       ,@(make-session-rows first-starttime timeslots-and-sessions))))
		     grouped-timeslots))
       (map cadr timeslots-and-sessions-with-selection))))

  (define (boolean-template name vf)
    `(,name ,vf
	    ,(readonly-vf 'no "no")
	    ,(readonly-vf 'yes "yes")))

  (define (card-expiry-year-metadata)
    (let ((current-year (date-year (time-monotonic->date (current-time 'time-monotonic)))))
      `((options ,@(map (lambda (x) (number->string (+ x current-year)))
			(iota 10))))))

  (define (registration-workflow)
    (let-values* (((registration) (make-db-result demo-app:registration))
		  ((hotel-date-options) (with-demo-app-transaction (get-hotel-date-options)))
		  ((registration-template) (ui-db-result registration
							 (build-all-parameters
							  `((rate_selection required1)
							    (hotelroom_guid required1)
							    (checkin ,hotel-date-options)
							    (checkout ,hotel-date-options)
							    (title ,(with-demo-app-transaction
								     (get-title-options)))
							    ))))
		  ((socialfunctions) (validated-field/parameters 'socialfunctions '() '()))
		  ((dietaryrequirements) (validated-field/parameters 'dietaryrequirements '() '()))
		  ((accommodation-required) (validated-field/parameters 'required
									'()
									'(required1)))
		  ((dataprotection) (validated-field/parameters 'dataprotection '() '()))
		  ((session-days session-selections)
		   (with-demo-app-transaction (get-session-blocks-for-ui)))
		  ((template-parts) (with-demo-app-transaction
				     (list (ui-db-result/ro (get-application-data))
					   registration-template
					   socialfunctions
					   (get-rate-groups-for-ui)
					   session-days
					   (boolean-template 'accommodation accommodation-required)
					   (boolean-template 'dataprotection dataprotection)
					   (get-hotels-for-ui)
					   dietaryrequirements
					   `(diet ,@(map ui-db-result/ro
							 (get-dietaryrequirements))))))
		  ((template) (apply make-xslt-ui-template template-parts)))

      (define (build-payment-template1 heading type want-cc-guarantee)
	`(payment ,(readonly-vf 'heading heading)
		  ,(readonly-vf 'type type)
		  (payment-method ,(validated-field/parameters 'chosen '()
							       '(required1))
				  ,(readonly-vf 'invoice "invoice")
				  ,(readonly-vf 'credit-card "credit-card"))
		  ,@(if want-cc-guarantee
			`((credit-card-guarantee ,(validated-field/parameters 'chosen '() '())
						 ,(readonly-vf 'yes "yes")))
			'())
		  ,(validated-field/parameters 'card-type '()
					       '((options "MasterCard"
							  "VISA")))
		  ,(validated-field/parameters 'card-number '()
					       '())
		  ,(validated-field/parameters 'card-expiry-month '()
					       '((options ("January" "1")
							  ("February" "2")
							  ("March" "3")
							  ("April" "4")
							  ("May" "5")
							  ("June" "6")
							  ("July" "7")
							  ("August" "8")
							  ("September" "9")
							  ("October" "10")
							  ("November" "11")
							  ("December" "12"))))
		  ,(validated-field/parameters 'card-expiry-year '()
					       (card-expiry-year-metadata))
		  ,(validated-field/parameters 'cardholder-name '()
					       '())))

      (define (accommodation-required?)
	(string=? (first (validated-field-content accommodation-required)) "yes"))

      (define (build-payment-template-parts)
	`((payments ,(build-payment-template1 "Conference payment" "conference" #f)
		    ,@(if (accommodation-required?)
			  (list (build-payment-template1 "Hotel payment" "hotel" #t))
			  '()))))

      (define (select-rate) (send-xslt-ui `((select-rate ())) template))
      (define (select-sessions) (send-xslt-ui `((select-sessions ())) template))
      (define (select-accommodation)
	(send-xslt-ui `((accommodation-required ())) template)
	(if (accommodation-required?)
	    (send-xslt-ui `((select-accommodation ())) template)
	    (begin
	      (db-result-set! registration 'hotelroom_guid '())
	      (db-result-set! registration 'checkin '())
	      (db-result-set! registration 'checkout '())
	      (db-result-set! registration 'specialrequirements_hotel "")
	      (db-result-set! registration 'companionname ""))))
      (define (select-dietaryrequirements)
	(send-xslt-ui `((select-dietaryrequirements ())) template))
      (define (enter-personal-details)
	(send-xslt-ui `((enter-personal-details ())) template
		      (lambda (action die)
			(when (null? (validated-field-content dataprotection))
			  (set-validated-field-error! dataprotection
						      (string-append
						       "Please confirm that you have read and "
						       "understood ... %%%"))
			  (die)))))
      (define (enter-payment-details payment-template-parts)
	(send-xslt-ui `((enter-payment-details ()))
		      (apply make-xslt-ui-template payment-template-parts)))

      (define (confirm-registration payment-template-parts)
	(let* ((extra-parts
		(with-demo-app-transaction
		 (let* ((rate (find-by-field 'guid (db-result-get registration 'rate_selection)
					     (get-registration_rates)))
			(rategroup (find-by-field 'position (db-result-get rate 'rate_group)
						  (get-registration_rate_groups)))
			(hotelroom (find-by-field 'guid (db-result-get registration
								       'hotelroom_guid)
						  (get-hotelrooms)))
			(hotel (and hotelroom
				    (find-by-field 'guid (db-result-get hotelroom 'hotel_guid)
						   (get-hotels)))))
		   `((reference ,(ui-db-result/ro rate)
				,(ui-db-result/ro rategroup)
				,@(if (accommodation-required?)
				      (list (ui-db-result/ro hotelroom)
					    (ui-db-result/ro hotel))
				      '())
				(dietaryrequirements
				 ,@(map (lambda (guid)
					  (ui-db-result/ro
					   (find-by-field 'guid guid (get-dietaryrequirements))))
					(validated-field-content dietaryrequirements))))))))
	       (confirmation-template (apply make-xslt-ui-template
					     (append template-parts
						     payment-template-parts
						     extra-parts))))
	  (send-xslt-ui `((confirm-registration ())) confirmation-template)))

      (define (submit-registration! payment-template-parts)
	(with-demo-app-transaction
	 (let ((registration-guid (db-table-save-result-pk1 (db-save! registration))))
	   (db-result-set! registration 'guid registration-guid)
	   (for-each (lambda (selection)
		       (db-result-set! selection 'registration_guid
				       registration-guid)
		       (db-save! selection #t))
		     session-selections)
	   (for-each (lambda (socialfunction-guid)
		       (db-save! (make-db-result demo-app:registration_socialfunction
						 'registration_guid registration-guid
						 'socialfunction_guid socialfunction-guid)
				 #t))
		     (validated-field-content socialfunctions))
	   (for-each (lambda (dietaryrequirement-guid)
		       (db-save! (make-db-result demo-app:registration_dietaryrequirement
						 'registration_guid registration-guid
						 'dietaryrequirement_guid
						   dietaryrequirement-guid)
				 #t))
		     (validated-field-content dietaryrequirements))
	   (for-each (lambda (payment-sxml)
		       (with-sxml (list payment-sxml)
			 (db-save! (make-db-result demo-app:registration_payment
						   'registration_guid registration-guid
						   'payment_type ($ "/payment/type")
						   'encrypteddetail
						     (xxexpr->string (list payment-sxml))
						   'processed #f))))
		     (template->plain-sxml
		      (with-sxml payment-template-parts (select "/payments/payment"))))))
	(send-xslt-ui/finish `((registration-complete ())) template))

      (let loop ()
	(select-rate)
	(select-sessions)
	(select-accommodation)
	(select-dietaryrequirements)
	(enter-personal-details)
	(let ((payment-template-parts (build-payment-template-parts)))
	  (enter-payment-details payment-template-parts)
	  (let confirmation-loop ()
	    (case (confirm-registration payment-template-parts)
	      ((edit-registration) (select-rate) (confirmation-loop))
	      ((edit-breakout-sessions) (select-sessions) (confirmation-loop))
	      ((edit-accommodation) (select-accommodation) (confirmation-loop))
	      ((edit-special-requirements) (select-dietaryrequirements) (confirmation-loop))
	      ((edit-personal-details) (enter-personal-details) (confirmation-loop))
	      ((edit-payment-details) (begin (enter-payment-details payment-template-parts)
					     (confirmation-loop)))
	      ((submit-registration) (submit-registration! payment-template-parts))
	      (else (loop))))))))

  (define-publication-patterns
    (home-pattern ()))

  (publish-pattern home-pattern ()
		   (registration-workflow)))
