(require-library 'sxml/sxml-match)
(require-library 'utils/sxml-tools)

(module utils/dom-tools
    (sxml->dom
     new-dom-document
     parse-xml-file)
  
  (import s2j)
  (import util/misc)
  (import sxml-accessors)
  (import utils/sxml-tools)
  (import string-io)
    
  ;; Tools for making DOM trees and maybe fooling about with them

  (define-java-classes
    (<document-builder-factory> |javax.xml.parsers.DocumentBuilderFactory|)
    (<java.io.File> |java.io.File|))

  (define-generic-java-methods
    new-instance
    set-namespace-aware
    set-validating
    new-document-builder
    new-document
    parse

    create-element
    (create-element-ns |createElementNS|)
    append-child
    create-text-node
    create-processing-instruction
    (get-namespace-uri |getNamespaceURI|)
    ;; create-attribute
    get-node-name
    get-prefix
    set-attribute
    (set-attribute-ns |setAttributeNS|)
    (jequals equals)
    concat)

  (define *XMLNS-NAMESPACE-URI* (->jstring "http://www.w3.org/2000/xmlns/"))

  ;; The only change is fdown is now typed:
  ;; fdown :: seed -> node -> (seed -> seed) -> seed
  ;;
  (define (foldts* fdown fup fhere seed tree)
    (cond
     ((null? tree) seed)
     ((not (pair? tree))		; An atom
      (fhere seed tree))
     (else
      (fdown seed tree
	     (lambda (kid-seed)
	       (let loop ((kid-seed kid-seed) (kids (cdr tree)))
		 (if (null? kids)
		     (fup seed kid-seed tree)
		     (loop (foldts* fdown fup fhere kid-seed (car kids))
			   (cdr kids)))))))))

  (define (create-document-builder namespace-aware validating)
    (let ((factory (new-instance (java-null <document-builder-factory>))))
      (set-namespace-aware factory (->jboolean namespace-aware))
      (set-validating factory (->jboolean validating))
      (new-document-builder factory)))

  (define (new-dom-document)
    (new-document (create-document-builder #f #f)))

  (define (parse-xml-file filename namespace-aware validating)
    (parse (create-document-builder namespace-aware validating)
	   (java-new <java.io.File> (->jstring filename))))

  (define (sxml->dom sxml)

    (define (any->string x)
      (cond
       ((string? x) x)
       ((number? x) (number->string x))
       ((symbol? x) (symbol->string x))
       (else (call-with-output-string (lambda (port) (write x port))))))

    (define genprefix
      (let ((count 0))
	(lambda ()
	  (set! count (+ count 1))
	  (->jstring (string-append "ns" (number->string count))))))

    (define (qname prefix tagname)
      (concat prefix (concat (->jstring ":") tagname)))

    (let ((doc (new-dom-document)))
      (foldts*
     
       ;; downwards: make a DOM node of where I am for kids to add themselves to,
       ;; and add to the seed
       (lambda (parent node process-kids)
	 (case (xml-element-tag node)
	   ((*top*) (process-kids parent))
	   ((*pi*)  (if (eq? (cadr node) 'xml)
			parent
			(begin
			  (append-child
			   parent
			   (create-processing-instruction doc
							  (->jstring (car (cdr node)))
							  (->jstring (apply string-append
									    (cdr (cdr node))))))
			  parent)))
	   ((|@|) (begin
		    (for-each (lambda (attr) (set-attribute
					      parent
					      (->jstring (car attr))
					      (->jstring (cadr attr))))
			      (cdr node))
		    parent))
	   (else
	    (let* ((tag (split-qname (car node)))
		   (new-node
		    (if (car tag)
			(let ((namespaceURI (->jstring (car tag)))
			      (parent-prefix (get-prefix parent))
			      (tagname (->jstring (cdr tag))))
			  (cond ((->boolean (jequals namespaceURI (get-namespace-uri parent)))
				 (create-element-ns doc
						    namespaceURI
						    (qname (if (java-null? parent-prefix)
							       (genprefix)
							       parent-prefix)
							   tagName)))
				(else
                                 (let* ((prefix (genprefix))
                                        (node
                                         (create-element-ns doc
                                                            namespaceURI
                                                            (qname prefix tagName))))
                                   (set-attribute-ns node *XMLNS-NAMESPACE-URI*
                                                     (qname (->jstring "xmlns") prefix) namespaceURI)
                                   node))))
			(create-element doc (->jstring (cdr tag))))))
	      (append-child parent new-node)
	      (process-kids new-node)))))

       ;; up: I don't need to do anything, because everything was added on the way down
       (lambda (parent last-kid node)
	 parent)
     
       ;; here: the parent node is threaded through as the seed
       (lambda (parent atom)
	 ;; create a textual node and add to the parent
	 (let ((txt (create-text-node doc (->jstring (any->string atom)))))
	   (append-child parent txt))
	 parent)

       ;; start with a document
       doc
       sxml)))

  )
