;-------------------------------------------------------------------------------
; Class hierarchy

(define-class <doc-module> <object> (classes variables functions macros))

(define-class <doc-object> <object> (module name title description))

(define-class <doc-class> <doc-object> (super subs ivars functions))
(define-class <doc-ivar> <doc-object> (class type))
(define-class <doc-variable> <doc-object> (type))
(define-class <doc-function> <doc-object> (kind class))
(define-class <doc-macro> <doc-object> ())

;-------------------------------------------------------------------------------
; Initializers

(define-method initialize <doc-module>
  (lambda (self)
    (set-classes! self '())
    (set-variables! self '())
    (set-functions! self '())
    (set-macros! self '())
    self))

(define-method initialize <doc-object>
  (lambda (self m n t d)
    (set-module! self m)
    (set-name! self n)
    (set-title! self t)
    (set-description! self d)
    self))

(define-method initialize <doc-class>
  (lambda (self m n t d sup)
    (set-super! self sup)
    (if sup
	(add-subclass! sup self))
    (set-subs! self '())
    (set-ivars! self '())
    (set-functions! self '())
    (add-class! 
    (next-method self m n t d)))

(define-method initialize <doc-ivar>
  (lambda (self m n t d c ty)
    (set-class! self c)
    (if c
	(add-ivar! c self))
    (set-type! self ty)
    (next-method self m n t d)))

(define-method initialize <doc-variable>
  (lambda (self m n t d ty)
    (set-type! self ty)
    (next-method self m n t d)))

(define-method initialize <doc-function>
  (lambda (self m n t d k c)
    (set-kind! self k)
    (set-class! self c)
    (if c
	(add-function! c self))
    (next-method self m n t d)))

;-------------------------------------------------------------------------------
; Methods on <doc-class>

(define-method add-subclass! <doc-class>
  (lambda (self sc)
    (set-subs! self (cons sc (subs self)))))

(define-method add-ivar! <doc-class>
  (lambda (self i)
    (set-ivars! self (cons i (ivars self)))))

(define-method add-function! <doc-class>
  (lambda (self f)
    (set-functions! self (cons f (functions self)))))

;-------------------------------------------------------------------------------
; Database management and toplevel interface

; Records structured as follows:
;	(subclass-of-<doc-object>-name
;		module-name
;		name-symbol title-string
;		(optional-field1 optional-field2 ...)
;	    .
;		description-items)

(define (process-record record-list)
  (let* ((class-name (car record-list))
	 (module-name (cadr record-list))
	 (item-name (caddr record-list))
	 (title-string (cadr (cddr record-list)))
	 (optional-fields (caddr (cddr record-list)))
	 (description-items (cdddr (cddr record-list)))

	 (the-class (symbol-value class-name))
	 (the-description (reduce + "" (map (lambda (x) ((compile x))) description-items))))

    (let ((the-doc-obj (apply make the-class module-name
			      		     item-name
					     title-string
					     the-description
					     optional-fields)))
      #t)))
