(define-class <avl-tree> <object> (tree-root
				   tree-less-than-predicate
				   tree-equality-predicate))

(define-class <avl-tree-node> <object> (node-left node-right node-value node-height))

(define-method initialize <avl-tree>
  (lambda (self lessp equalp)
    (set-tree-less-than-predicate! self lessp)
    (set-tree-equality-predicate! self equalp)
    (set-tree-root! self #f)
    self))

(define-method initialize <avl-tree-node>
  (lambda (self value)
    (set-node-value! self value)
    (set-node-height! self 1)
    (set-node-left! self #f)
    (set-node-right! self #f)
    self))

(define (branch-height branch)
  (if branch
      (node-height branch)
      0))

(define (update-node-height! node)
  (set-node-height! node (+ (max (branch-height (node-left node))
				 (branch-height (node-right node)))
			    1)))

(define (rotate-left! node)
  (let* ((r (node-right node))
	 (b (node-left r)))
    (set-node-right! node b)
    (set-node-left! r node)
    (update-node-height! node)
    (update-node-height! r)
    r))

(define (rotate-right! node)
  (let* ((l (node-left node))
	 (b (node-right l)))
    (set-node-left! node b)
    (set-node-right! l node)
    (update-node-height! node)
    (update-node-height! l)
    l))

(define (rebalance-left! node)
  (if (>= (+ (branch-height (node-right node)) 1)
	  (branch-height (node-left node)))
      (begin
	(update-node-height! node)
	node)
      (let ((q (node-left node)))
	(if (<= (branch-height (node-right q))
		(branch-height (node-left q)))
	    (rotate-right! node)
	    (begin
	      (set-node-left! node (rotate-left! q))
	      (rotate-right! node))))))

(define (rebalance-right! node)
  (if (>= (+ (branch-height (node-left node)) 1)
	  (branch-height (node-right node)))
      (begin
	(update-node-height! node)
	node)
      (let ((q (node-right node)))
	(if (<= (branch-height (node-left q))
		(branch-height (node-right q)))
	    (rotate-left! node)
	    (begin
	      (set-node-right! node (rotate-right! q))
	      (rotate-left! node))))))

(define-method insert <avl-tree>
  (lambda (self value)
    (let ((newnode (make <avl-tree-node> value))
	  (less? (tree-less-than-predicate self)))
      (set-tree-root! self
	(let walk ((node (tree-root self)))
	  (cond
	   ((not node)
	    newnode)
	   ((less? value (node-value node))
	    (set-node-left! node (walk (node-left node)))
	    (rebalance-left! node))
	   ;;; Allows duplicate entries.
	   (else
	    (set-node-right! node (walk (node-right node)))
	    (rebalance-right! node))))))))

(define-method lookup <avl-tree>
  (lambda (self value)
    (let ((equalp? (tree-equality-predicate self))
	  (lessp? (tree-less-than-predicate self)))
      (let walk ((node (tree-root self)))
	(cond
	 ((not node)
	  #f)
	 ((equalp? value (node-value node))
	  node)
	 ((lessp? value (node-value node))
	  (walk (node-left node)))
	 (else
	  (walk (node-right node))))))))

(define-method as-list <avl-tree>
  (lambda (self)
    (let convert ((node (tree-root self)) (acc '()))
      (if node
	  (convert (node-left node)
		   (cons (node-value node)
			 (convert (node-right node) acc)))
	  acc))))

(define-method print-string <avl-tree>
  (lambda (self w?)
    (+ "#<"
       (let p ((l (as-list self)))
	 (if (null? l)
	     ">"
	     (+ (print-string (car l) #t)
		(if (null? (cdr l))
		    ">"
		    (+ ", " (p (cdr l))))))))))

(define-method print-string <avl-tree-node>
  (lambda (self w?)
    (+ "#node["
       (+ (print-string (node-value self) #t)
	  (+ "]["
	     (+ (print-string (node-height self) #t) "]"))))))
