;;; STARTUP.M   Copyright (C) 1994 Tony Garnock-Jones
;;
;;  Initialisation code for MOOF.
;;

;-------------------------------------------------------------------------
; Definitions needed for macro-expansion code to load and run correctly.

(define list (lambda x x))

(define map
  (lambda (f l)
    (if (null? l)
	'()
	(cons (f (car l)) (map f (cdr l))))))

(define caar (lambda (x) (car (car x))))
(define cadr (lambda (x) (car (cdr x))))
(define cdar (lambda (x) (cdr (car x))))
(define cddr (lambda (x) (cdr (cdr x))))
(define caaar (lambda (x) (car (car (car x)))))
(define caadr (lambda (x) (car (car (cdr x)))))
(define cadar (lambda (x) (car (cdr (car x)))))
(define caddr (lambda (x) (car (cdr (cdr x)))))
(define cdaar (lambda (x) (cdr (car (car x)))))
(define cdadr (lambda (x) (cdr (car (cdr x)))))
(define cddar (lambda (x) (cdr (cdr (car x)))))
(define cdddr (lambda (x) (cdr (cdr (cdr x)))))

;-------------------------------------------------------------------------

(or (load "macro.m")
    (load (+ *library-directory* "/macro.m"))
    (raise-exception 'macro-library-not-found "macro.m"))

; Make sure the rest of the file is macro-expanded
(for-each1 (lambda (expr) ((compile (macro-expand-pre-eval expr)))) '(

;-------------------------------------------------------------------------

(define (vector->list vec)
  (let ((len (indexed-length vec)))
    (let loop ((i 0))
      (if (>= i len)
	  '()
	  (cons (indexed-ref vec i) (loop (+ i 1)))))))

(define (list->vector lst)
  (let* ((len (list-length lst))
	 (vec (make <vector> len)))
    (let loop ((i 0) (lst lst))
      (if (null? lst)
	  vec
	  (begin
	    (indexed-set! vec i (car lst))
	    (loop (+ i 1) (cdr lst)))))))

(define-macro (quasiquote exp)
  (let xform ((exp exp) (depth 1))
    (cond
     ((instance? exp <pair>)
      (case (car exp)
	((quasiquote) (list 'list ''quasiquote (xform (cadr exp) (+ depth 1))))
	((unquote) (if (= depth 1)
		       (cadr exp)
		       (list 'list ''unquote (xform (cadr exp) (- depth 1)))))
	(else (if (and (instance? (car exp) <pair>)
		       (eq? (caar exp) 'unquote-splicing))
		  (if (= depth 1)
		      (list '+ (cadar exp) (xform (cdr exp) depth))
		      (list 'cons
			    (xform (car exp) (- depth 1))
			    (xform (cdr exp) depth)))
		  (list 'cons
			(xform (car exp) depth)
			(xform (cdr exp) depth))))))
     ((instance? exp <vector>)
      (list 'list->vector (xform (vector->list exp) depth)))
     (else
      (list 'quote exp)))))

;-------------------------------------------------------------------------

(define-macro (do bindings term . exps)
  `(let %%doloop ,(map (lambda (binding) `(,(car binding) ,(cadr binding)))
		       bindings)
     (if ,(car term)
	 (begin ,@(cdr term))
	 (begin ,@exps
		(%%doloop ,@(map caddr bindings))))))

;-------------------------------------------------------------------------

(define vector
  (lambda args
    (let ((vec (make <vector> (list-length args))))
      (do ((i 0 (+ i 1))
	   (l args (cdr l)))
	  ((null? l) vec)
	(indexed-set! vec i (car l))))))

(let ((set-ivarnames! (compute-slot-setter <class> 'instance-variable-names)))
  (set-ivarnames! <symbol> '("Invalid Instance-variable Name" global-value))
  (set-ivarnames! <pair> '(car cdr))
  (set-ivarnames! <function> '(env argument-count name))
  (set-ivarnames! <primitive-function> '(argument-count name))
  (set-ivarnames! <generic-function> '(name methods))
  (set-ivarnames! <thread>
		  '(continuation ip env acc func curr-gf stack-depth quantum
				 status next prev handler-chain))
  (set-ivarnames! <continuation>
		  '(partial-continuation stack handler-chain))
  (set-ivarnames! <stream> '())
  (set-ivarnames! <string-stream> '(string pos cache))
  (set-ivarnames! <file-stream> '(name))
  (if (provided? 'sockets)
      (set-ivarnames! <socket> '(number))))

(let ((set-gf-name! (compute-slot-setter <generic-function> 'name))
      (set-gf-methods! (compute-slot-setter <generic-function> 'methods)))
  
  ; The reason this isn't a (define) is that (define) is not valid except
  ; at the toplevel. Using define-global-variable is not as pretty to read,
  ; but avoids the problems.
  (define-global-variable 'make
    (lambda (class . args)
      (apply initialize (simple-make class 0 0) args)))

  (add-method initialize (list <generic-function>)
	      (lambda (self name)
		(set-gf-name! self name)
		(set-gf-methods! self '())
		self)))

(define-macro (define-generic name)
  `(define-generic-variable ',name))

(define (define-generic-variable name)
  (if (not (global-variable-bound? name))
      (define-global-variable name
	(make <generic-function> name))))

(define-macro (define-method gf class method)
  (if (instance? gf <symbol>)
      `(begin
	 (define-generic ,gf)
	 (add-method ,gf (list ,class) ,method))
      `(add-method ,gf (list ,class) ,method)))

(define (assert-type! x type)
  (or (instance? x type)
      (raise-exception 'failed-type-assertion (list x type))))

(define-method string-length <string>
  (lambda (self)
    (- (binary-length self) 1)))

(define-method stream-print <object>
  (lambda (self stream)
    (let ((s (print-string self #f)))
      (write-chars-to stream (string-length s) s))))

(define-method stream-write <object>
  (lambda (self stream)
    (let ((s (print-string self #t)))
      (write-chars-to stream (string-length s) s))))

(define-method print <object>
  (lambda (self)
    (stream-print self standard-output)
    (stream-flush standard-output)))

(define-method write <object>
  (lambda (self)
    (stream-write self standard-output)
    (stream-flush standard-output)))

(define (newline) (print "\n"))

(define-method initialize <pair>
  (lambda (self a d)
    (set-car! self a)
    (set-cdr! self d)
    self))

(define-method initialize <vector>
  (lambda (self size . args)
    (let ((new-vector (simple-make <vector> size 0)))
      (if (not (null? args))
	  (let do-next-elt ((n 0))
	    (if (< n size)
		(begin
		  (indexed-set! new-vector n (car args))
		  (do-next-elt (+ n 1))))))
      new-vector)))

(define for-each for-each1)	; Temporary measure.

(define (for-each-vector func vec)
  (let ((idx 0)
	(max (indexed-length vec)))
    (while (< idx max)
	   (func (indexed-ref vec idx))
	   (set! idx (+ idx 1))))
  '())

(define-method symbol-value <symbol>
  (compute-slot-getter <symbol> 'global-value))

(define-method initialize <class>
  (let ((set-ivarnames!
	 (compute-slot-setter <class> 'instance-variable-names))
	(set-super! (compute-slot-setter <class> 'super-class))
	(ivarcount (compute-slot-getter <class> 'instance-variable-count))
	(set-ivarcount!
	 (compute-slot-setter <class> 'instance-variable-count)))
    (lambda (self super ivarnames)
      (set-super! self super)
      (set-ivarnames! self ivarnames)
      (set-ivarcount! self
		      (+ (list-length ivarnames) (ivarcount super)))
      self)))

(define-macro (define-class name super ivars)
  `(let* ((class
	   (make <class> ,super
		 (map
		  (lambda (ivar)
		    (if (instance? ivar <pair>)
			(car ivar)
			ivar))
		  ',ivars)))
	  (do-getter
	   (lambda (name type)
	     (cond
	      ((eq? type #t)
	       (define-generic-variable name)
	       (define-method (symbol-value name) class
		 (compute-slot-getter class name)))
	      ((eq? type #f))
	      (else
	       (define-generic-variable type)
	       (define-method (symbol-value type) class
		 (compute-slot-getter class name))))))
	   (do-setter
	    (lambda (name type)
	      (let ((proc (as-symbol (+ "set-"
					(+ (as-string name) "!")))))
		(cond
		 ((eq? type #t)
		  (define-generic-variable proc)
		  (define-method (symbol-value proc) class
		    (compute-slot-setter class name)))
		 ((eq? type #f))
		 (else
		  (define-generic-variable type)
		  (define-method (symbol-value type) class
		    (compute-slot-setter class name))))))))
      (define-global-variable ',name class)
      (for-each
       (lambda (ivar)
	 (if (instance? ivar <pair>)
	     (let ((name (car ivar)) (type (cdr ivar)))
	       (cond
		((null? type)
		 (do-getter name #t)
		 (do-setter name #t))
		((null? (cdr type))
		 (do-getter name (car type))
		 (do-setter name #t))
		(else
		 (do-getter name (car type))
		 (do-setter name (car (cdr type))))))
	     (begin
	       (do-getter ivar #t)
	       (do-setter ivar #t))))
       ',ivars)))

(define (read) (read-from standard-input))

(define-method next-thread <thread>
  (compute-slot-getter <thread> 'next))
(define-method next-thread <null>
  (lambda (null)
    (current-thread)))

(define-method equal? <pair>
  (lambda (a b)
    (and (instance? b <pair>)
	 (equal? (car a) (car b))
	 (equal? (cdr a) (cdr b)))))

(define-method equal? <vector>
  (lambda (a b)
    (let ((i1 (indexed-length a))
	  (i2 (indexed-length b)))
      (if (eq? i1 i2)
	  (letrec ((l (lambda (n)
			(cond
			 ((eq? n i1) #t)
			 ((equal? (indexed-ref a n)
				  (indexed-ref b n))
			  (l (+ n 1)))
			 (else #f)))))
	    (l 0))
	  #f))))

(define-method print-string <class>
  (let ((instvar (compute-slot-getter <class> 'instance-variable-names)))
    (lambda (self w)
      (+ "#<class "
	 (+ (print-string (instvar self) #f)
	    ">")))))

(define-method print-string <pair>
  (letrec ((depth 0)
	   (rec
	    (lambda (rest w)
	      (cond
	       ((> depth 64) " ...")
	       ((instance? rest <pair>)
		(set! depth (+ depth 1))
		(let ((r (+ " " (+ (print-string (car rest) w) (rec (cdr rest) w)))))
		  (set! depth (- depth 1))
		  r))
	       ((null? rest) ")")
	       (else
		(+ " . " (+ (print-string rest w) ")")))))))
    (lambda (self w)
      (if (> depth 64)
	  "(...)"
	  (begin
	    (set! depth (+ depth 1))
	    (let ((r (+ "(" (+ (print-string (car self) w) (rec (cdr self) w)))))
	      (set! depth (- depth 1))
	      r))))))

(define-method print-string <vector>
  (letrec ((depth 0)
	   (rec
	    (lambda (vec idx max w)
	      (if (eq? idx max)
		  ")"
		  (+ " " (+ (print-string (indexed-ref vec idx) w) (rec vec (+ idx 1) max w)))))))
    (lambda (self w)
      (if (> depth 64)
	  "#(...)"
	  (let ((max (indexed-length self)))
	    (set! depth (+ depth 1))
	    (let ((r (+ "#("
			(if (eq? max 0)
			    ")"
			    (+ (print-string (indexed-ref self 0) w) (rec self 1 max w))))))
	      (set! depth (- depth 1))
	      r))))))

(define-method print-string <function>
  (let ((argc (compute-slot-getter <function> 'argument-count))
	(name (compute-slot-getter <function> 'name)))
    (lambda (self w)
      (+ "#<function "
	 (+ (if (not (null? (name self)))
		(+ (print-string (name self) #f) " ")
		"")
	    (+ (print-string (argc self) #f) ">"))))))

(define-method print-string <primitive-function>
  (let ((argc (compute-slot-getter <primitive-function> 'argument-count))
	(name (compute-slot-getter <primitive-function> 'name)))
    (lambda (self w)
      (+ "#<primitive "
	 (+ (print-string (name self) #f)
	    (+ " "
	       (+ (print-string (argc self) #f) ">")))))))

(define-method hash-for-equal <pair>
  (lambda (self)
    (binary-xor (hash-for-equal (car self))
		(hash-for-equal (cdr self)))))

(define-method hash-for-equal <vector>
  (lambda (self)
    (let loop ((index 0) (hash 0))
      (if (< index (indexed-length self))
	  (loop (+ index 1) (binary-xor hash
					(hash-for-equal
					 (indexed-ref self index))))
	  hash))))

(define-method gf-name <generic-function>
  (compute-slot-getter <generic-function> 'name))

(define-method gf-methods <generic-function>
  (compute-slot-getter <generic-function> 'methods))

(define-method print-string <generic-function>
  (lambda (self w)
    (+ "#<generic-function "
       (+ (print-string (gf-name self) #f) ">"))))

(define-method partial-continuation <continuation>
  (compute-slot-getter <continuation> 'partial-continuation))

(define-method continuation-stack <continuation>
  (compute-slot-getter <continuation> 'stack))

(define-method rebuild-continuation <continuation>
  (let ((sp! (compute-slot-setter <continuation> 'partial-continuation))
	(ss! (compute-slot-setter <continuation> 'stack))
	(sh! (compute-slot-setter <continuation> 'handler-chain))
	(gh (compute-slot-getter <continuation> 'handler-chain)))
    (lambda (self partial-continuation)
      (let ((c (simple-make <continuation> 0 0)))
	(sp! c partial-continuation)
	(ss! c (continuation-stack self))
	(sh! c (gh self))
	c))))
      
(define-method print-string <continuation>
  (lambda (self w)
    (+ "#<continuation "
       (+ (print-string (continuation-stack self) #t) ">"))))

(define-method min <object>
  (lambda (a b)
    (if (< a b)
	a
	b)))

(define-method max <object>
  (lambda (a b)
    (if (> a b)
	a
	b)))

(define (member what in)
  (cond
   ((null? in) #f)
   ((equal? (car in) what) in)
   (else
    (member what (cdr in)))))

(define (assoc o a)
  (cond
   ((null? a) #f)
   ((equal? (caar a) o) (car a))
   (else
    (assoc o (cdr a)))))

(define (reduce f seed l)
  (if (null? l)
      seed
      (reduce f (f seed (car l)) (cdr l))))

;-------------------------------------------------------------------------

(define-class <eof-object> <object> ())

(define-method print-string <eof-object>
  (lambda (self w)
    "#<eof-object>"))

(define-method initialize <eof-object>
  (let ((eof-object (simple-make <eof-object> 0 0)))
    (lambda (self)
      eof-object)))

(define *eof-object*
  (let ((eof-object (make <eof-object>)))
    (lambda ()
      eof-object)))

(define (eof-object? obj)
  (instance? obj <eof-object>))

(define %%the-eof-object (*eof-object*))

;-------------------------------------------------------------------------

(define (enter-critical-section thunk)
  (set-critical-section-flag! #t)
  (thunk)
  (set-critical-section-flag! #f))

;-------------------------------------------------------------------------

(define (*undefined*) (begin))

;-------------------------------------------------------------------------

(define load
  (let ((try-load macro-expanding-primitive-load))
    (lambda (filename)
      (or (try-load filename)
	  (try-load (+ *library-directory* (+ "/" filename)))
	  (try-load (+ filename ".m"))
	  (try-load (+ *library-directory* (+ "/" (+ filename ".m"))))
	  (raise-exception 'could-not-open filename)))))

(load "require.m")

(define (try handler thunk)
  (install-handler handler)
  (let ((result (thunk)))
    (remove-handler)
    result))

(define *signals* '#(nullsignal
		     sighup sigint sigquit sigill sigtrap sigiot sigbus sigfpe
		     sigkill sigusr1 sigsegv sigusr2 sigpipe sigalrm sigterm sigchld
		     sigcont sigstop sigtstp sigttin sigttou sigurg sigxcpu sigxfsz
		     sigvtalrm sigprof sigwinch sigio sigpwr))

(install-handler
 (lambda (ex args cont)
   (if (eq? ex 'signal-raised)
       (print "\nInterrupt received from user\n")
       (begin
	 (newline)
	 (print
"=========================================================================\n")
	 (print "SYSTEM-DEFAULT EXCEPTION HANDLER\n")
	 (print
"=========================================================================\n")
	 (print "Exception: ")
	 (print ex)
	 (print " (args: ")
	 (print args)
	 (print ")")
	 (newline)
	 (print "Continuation: ")
	 (print cont)
	 (newline)
	 (abort)))))

(define (abort)
  (block-thread (current-thread)))
(define quit abort)

(fork
 (lambda ()
   (enter-critical-section
    (lambda ()
      (let ((load-repl #t))
	(for-each-vector
	 (lambda (arg)
	   (cond
	    ((equal? arg "-r")
	     (set! load-repl #f)
	     (verbose #f))
	    ((equal? arg "-v")
	     (verbose #f))
	    (else (require arg))))
	 *command-line*)
	(if load-repl
	    (require 'repl)))))))

; Finish macro-expanding this file.
))
