/* $Id: interp.c,v 1.8 1998/04/05 10:33:48 tonyg Exp $ */

#include "memory.h"
#include "class.h"
#include "interp.h"
#include "vector.h"
#include "thread.h"
#include "pair.h"
#include "function.h"
#include "bytecode.h"
#include "symbol.h"
#include "prim.h"
#include "gf.h"

#include <string.h>

int interp_critsec;

PRIVATE OBJECT prepare_environment(OBJECT thread, int given_argc,
                                    int wanted_argc, OBJECT func) {
    OBJECT env = newvector(wanted_argc < 0 ? -wanted_argc : wanted_argc, 0);

    temp_register(&env, 1);

    if (wanted_argc < 0) {
        OBJECT lst = NULL;

        temp_register(&lst, 1);

        while (given_argc >= -wanted_argc) {
            lst = cons(stack_pop(thread), lst);
            given_argc--;
        }

        deregister_root(1);

        ISET(env, given_argc, lst);
    } else if (given_argc != wanted_argc) {
        raise_exception("wrong-argument-count", func);

        deregister_root(1);
        return NULL;
    }

    while (given_argc > 0) {
        ISET(env, given_argc - 1, stack_pop(thread));
        given_argc--;
    }

    deregister_root(1);
    return env;
}

PRIVATE OBJECT primitive_call(OBJECT thread, OBJECT prim, int argc) {
    OBJECT a[14];
    int wanted_argc = (int) NUM(GET(prim, PRIM_ARGC));

    memset(a, 0, sizeof(OBJECT) * 14);
    temp_register(a, 14);

    if (wanted_argc < 0) {
        OBJECT lst = NULL;

        temp_register(&lst, 1);
        wanted_argc = -wanted_argc;

        while (argc >= wanted_argc) {
            lst = cons(stack_pop(thread), lst);
            argc--;
        }

        deregister_root(1);

        stack_push(thread, lst);
        argc++;
    }

    if (argc != wanted_argc) {
	deregister_root(1);
        raise_exception("wrong-argument-count", prim);
        return NULL;
    }

    switch (wanted_argc) {
        case 14: a[13] = stack_pop(thread); /* FALL THROUGH */
        case 13: a[12] = stack_pop(thread); /* FALL THROUGH */
        case 12: a[11] = stack_pop(thread); /* FALL THROUGH */
        case 11: a[10] = stack_pop(thread); /* FALL THROUGH */
        case 10:  a[9] = stack_pop(thread); /* FALL THROUGH */
        case  9:  a[8] = stack_pop(thread); /* FALL THROUGH */
        case  8:  a[7] = stack_pop(thread); /* FALL THROUGH */
        case  7:  a[6] = stack_pop(thread); /* FALL THROUGH */
        case  6:  a[5] = stack_pop(thread); /* FALL THROUGH */
        case  5:  a[4] = stack_pop(thread); /* FALL THROUGH */
        case  4:  a[3] = stack_pop(thread); /* FALL THROUGH */
        case  3:  a[2] = stack_pop(thread); /* FALL THROUGH */
        case  2:  a[1] = stack_pop(thread); /* FALL THROUGH */
        case  1:  a[0] = stack_pop(thread); /* FALL THROUGH */
        default:
	  a[0] = getfunc(prim)(a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7],
			       a[8], a[9], a[10], a[11], a[12], a[13]);
	  deregister_root(1);
	  return a[0];
    }
}

PRIVATE OBJECT continuation_call(OBJECT thread, OBJECT cont, int argc) {
    OBJECT val = NULL;

    if (argc != 1) {
        raise_exception("wrong-argument-count", cont);
        return NULL;
    }

    temp_register(&val, 1);

    val = stack_pop(thread);

    setcont(thread, cont);
    pop_call(thread);

    deregister_root(1);
    return val;
}

OBJECT interp_apply(OBJECT func, OBJECT args) {
    OBJECT thread = getcurrthr();
    int given_argc = 0;
    OBJECT fclass = GETCLASS(func);
    OBJECT curr_gf = NULL;

    while (args != NULL)
        if ((CDR(args) == NULL) &&
            (CAR(args) == NULL || instance(CAR(args), pair_class))) {
                args = CAR(args);
                while (args != NULL) {
                    stack_push(thread, CAR(args));
                    given_argc++;
                    args = CDR(args);
                }
        } else {
            stack_push(thread, CAR(args));
            given_argc++;
            args = CDR(args);
        }

    if (subclass(fclass, gf_class)) {
        OBJECT method = matchmethods(
                            func,
                            thread,
                            stack_depth(thread) - given_argc,
                            NULL);

        if (method == NULL) {
            raise_exception("method-not-found", func);
            return NULL;
        }

        curr_gf = func;
        func = method;
        fclass = GETCLASS(func);
    }

    if (subclass(fclass, function_class)) {
        OBJECT frame = prepare_environment(thread, given_argc,
                                           (int) NUM(GET(func, FUNC_ARGC)),
                                           func);

        if (frame != NULL)
            push_call(thread, func, curr_gf,
                      cons(frame, GET(func, FUNC_ENV)));

        return NULL;
    } else if (subclass(fclass, prim_class))
        return primitive_call(thread, func, given_argc);
    else if (subclass(fclass, continuation_class))
        return continuation_call(thread, func, given_argc);
    else {
        raise_exception("applied-non-function", func);
        return NULL;
    }
}

#define EXTRACT_i16(buf, pos)	((signed short) (((unsigned short) buf[pos+1] << 8) + \
						 ((unsigned short) buf[pos])))

PRIVATE void disasm_1(OBJECT func, unsigned char *ip) {
    switch (*ip++) {
        case OP_LITREF:     printf("ldc     %d\n", *ip++); break;
        case OP_LOCALREF:   printf("ldl     %d %d\n", *ip, *(ip+1));
                            ip += 2; break;
        case OP_SLOTREF:    printf("lds     %d\n", *ip++); break;
        case OP_GLOBALREF:  printf("ldg     %d (%.40s) %p\n",
                                *ip,
                                BIDX(IGET(func, *ip), 0),
                                GET(IGET(func, *ip), SYM_VALUE)); ip++; break;

        case OP_LOCALSET:   printf("stl     %d %d\n", *ip, *(ip+1));
                            ip += 2; break;
        case OP_LOCALSET_NAMED:
                            printf("stln    %d %d (.40%s)\n",
                                *ip,
                                *(ip+1),
                                BIDX(IGET(func, *(ip+2)), 0));
                            ip += 3; break;
        case OP_SLOTSET:    printf("sts     %d\n", *ip++); break;
        case OP_GLOBALSET:  printf("stg     %d\n", *ip++); break;

        case OP_PUSH:       printf("push\n"); break;
        case OP_POP:        printf("pop\n"); break;

        case OP_ENTERSCOPE: printf("enter   %d\n", *ip++); break;
        case OP_EXITSCOPE:  printf("exit\n"); break;

        case OP_JUMP:       printf("j       %d\n", EXTRACT_i16(ip, 0));
                            ip += 2; break;
        case OP_JUMP_FALSE: printf("jf      %d\n", EXTRACT_i16(ip, 0));
                            ip += 2; break;
        case OP_JUMP_TRUE:  printf("jt      %d\n", EXTRACT_i16(ip, 0));
                            ip += 2; break;
        case OP_JUMP_EQ:    printf("je      %d\n", EXTRACT_i16(ip, 0));
                            ip += 2; break;

        case OP_CALL:       printf("call    %d\n", *ip++); break;
        case OP_TAIL_CALL:  printf("tail    %d\n", *ip++); break;
        case OP_CALL_NEXTMETHOD:    printf("callnm  %d\n", *ip++);
                                    break;
        case OP_TAIL_CALL_NEXT:     printf("tailnm  %d\n", *ip++);
                                    break;
        case OP_RETURN:     printf("ret\n"); break;

        case OP_MAKE_LAMBDA:    printf("lambda  %d\n", *ip++); break;
    }
}

OBJECT interp(OBJECT thread, int ticks) {
    OBJECT acc = NULL;

    temp_register(&thread, 1);
    temp_register(&acc, 1);

    acc = GET(thread, THR_ACC);

    while (ticks > 0 || ticks == -1 || interp_critsec) {
        OBJECT func, cls;

        func = GET(thread, THR_FUNC);

        if (signal_raised != NULL) {
            SET(thread, THR_ACC, acc);
            raise_exception("signal-raised", signal_raised);
            signal_raised = NULL;
            continue;
        }

        if (func == NULL) {
            SET(thread, THR_ACC, acc);
            deregister_root(2);
            return true;
        }

        cls = GETCLASS(func);

        if (subclass(cls, function_class)) {
            unsigned char *buf = (unsigned char *) BIDX(func, 0);
            word pos = (word) NUM(GET(thread, THR_IP));
            int should_save_state = 1;

            while (ticks > 0 || ticks == -1 || interp_critsec) {
                if (ticks > 0 && !interp_critsec)
                    ticks--;

                if (debug_flag > 1) {
		  switch (buf[pos]) {
		    default:
		      if (debug_flag == 2)
			break;

		    case OP_CALL:
		    case OP_TAIL_CALL:
		    case OP_CALL_NEXTMETHOD:
		    case OP_TAIL_CALL_NEXT:
		      printf("%.5d %.5d ", ticks, pos);
		      disasm_1(func, buf+pos);
		  }
                }

                switch (buf[pos++]) {
                    case OP_LITREF:
                        acc = IGET(func, buf[pos++]);
                        continue;

                    case OP_LOCALREF: {
                        word num = (word) buf[pos++];
                        OBJECT vec = GET(thread, THR_ENV);

                        while (num > 0) {
                            vec = CDR(vec);
                            num--;
                        }

                        vec = CAR(vec);
                        acc = IGET(vec, buf[pos++]);
                        continue;
                    }

                    case OP_SLOTREF:
                        acc = GET(acc, buf[pos++]);
                        continue;

                    case OP_GLOBALREF:
                        acc = GET(IGET(func, buf[pos++]), SYM_VALUE);

                        if (acc == undefined) {
			  if (should_save_state) {
			    SET(thread, THR_IP, MKNUM(pos));
			    should_save_state = 0;
			  }
			  raise_exception("referenced-undefined-global",
					  IGET(func, buf[pos-1]));
			  break;
                        }

                        continue;

                    case OP_LOCALSET: {
                        word num = (word) buf[pos++];
                        OBJECT vec = GET(thread, THR_ENV);

                        while (num > 0) {
                            vec = CDR(vec);
                            num--;
                        }

                        vec = CAR(vec);
                        ISET(vec, buf[pos++], acc);
                        continue;
                    }

                    case OP_LOCALSET_NAMED: {
                        word num = (word) buf[pos++];
                        OBJECT vec = GET(thread, THR_ENV);

                        while (num > 0) {
                            vec = CDR(vec);
                            num--;
                        }

                        vec = CAR(vec);
                        ISET(vec, buf[pos++], acc);

                        if (instance(acc, function_class))
                            SET(acc, FUNC_NAME, IGET(func, buf[pos++]));
                        else
                            pos++;

                        continue;
                    }

                    case OP_SLOTSET: {
                        OBJECT obj = stack_pop(thread);

                        SET(obj, buf[pos++], acc);
                        continue;
                    }

                    case OP_GLOBALSET: {
                        OBJECT sym = IGET(func, buf[pos++]);

                        if (GET(sym, SYM_VALUE) == undefined) {
			  if (should_save_state) {
			    SET(thread, THR_IP, MKNUM(pos));
			    should_save_state = 0;
			  }
			  raise_exception("set-undefined-global",
					  newvector(2, 1, sym, acc));
			  break;
                        }

                        SET(sym, SYM_VALUE, acc);
                        continue;
                    }

                    case OP_PUSH:
                        stack_push(thread, acc);
                        continue;

                    case OP_POP:
                        acc = stack_pop(thread);
                        continue;

                    case OP_ENTERSCOPE:
                        SET(thread, THR_ENV,
                            cons(newvector(buf[pos++], 0),
                                GET(thread, THR_ENV)));
                        continue;

                    case OP_EXITSCOPE:
                        SET(thread, THR_ENV, CDR(GET(thread, THR_ENV)));
                        continue;

                    case OP_JUMP:
			pos += EXTRACT_i16(buf, pos);
			continue;

                    case OP_JUMP_FALSE:
                        if (acc == false) {
			  pos += EXTRACT_i16(buf, pos);
                        } else
			  pos += 2;   /* Skip over relative address */

                        continue;

                    case OP_JUMP_TRUE:
                        if (acc != false) {
			  pos += EXTRACT_i16(buf, pos);
                        } else
			  pos += 2;   /* Skip over relative address */

                        continue;

                    case OP_JUMP_EQ:
                        if (acc == stack_pop(thread)) {
			  pos += EXTRACT_i16(buf, pos);
                        } else
			  pos += 2;   /* Skip over relative address */

                        continue;

                    case OP_TAIL_CALL:
                        pop_call(thread);
                        should_save_state = 0;

                        /* FALL THROUGH */

                    case OP_CALL: {
                        char argc = buf[pos++];
                        OBJECT class = GETCLASS(acc);   /* Acc is function */
                        OBJECT curr_gf = NULL;

                        if (should_save_state) {
                            SET(thread, THR_IP, MKNUM(pos));
                            should_save_state = 0;
                        }

                        if (subclass(class, gf_class)) {
                            OBJECT method = matchmethods(
                                                acc,
                                                thread,
                                                stack_depth(thread) - argc,
                                                NULL);

                            if (method == NULL) {
                                raise_exception("method-not-found", acc);
                                break;
                            }

                            curr_gf = acc;
                            acc = method;
                            class = GETCLASS(acc);
                        }

                        if (subclass(class, function_class)) {
                            OBJECT frame = prepare_environment(thread, argc,
                                            (int) NUM(GET(acc, FUNC_ARGC)),
                                            acc);

                            if (frame != NULL) {
			      OBJECT newenv = NULL;

			      temp_register(&newenv, 1);
			      newenv = frame;
			      newenv = cons(frame, GET(acc, FUNC_ENV));
			      push_call(thread, acc, curr_gf, newenv);
			      deregister_root(1);
			    }
                        } else if (subclass(class, prim_class))
                            acc = primitive_call(thread, acc, argc);
                        else if (subclass(class, continuation_class))
                            acc = continuation_call(thread, acc, argc);
                        else
                            raise_exception("applied-non-function", acc);

                        break;
                    }

                    case OP_TAIL_CALL_NEXT:
			acc = cons(GET(thread, THR_CURR_GF),
				   GET(thread, THR_FUNC));
                        pop_call(thread);
                        should_save_state = 0;

                        /* FALL THROUGH */

                    case OP_CALL_NEXTMETHOD: {
                        char argc = buf[pos++];
                        OBJECT method;
                        OBJECT class;

                        if (should_save_state) {
                            SET(thread, THR_IP, MKNUM(pos));
                            should_save_state = 0;
                        }

                        if (buf[pos - 2] == OP_CALL_NEXTMETHOD)
			  acc = cons(GET(thread, THR_CURR_GF),
				     GET(thread, THR_FUNC));
			/* otherwise, acc has already been set to info about the
			   current call-frame by OP_TAIL_CALL_NEXT just above. */

                        if (CAR(acc) == NULL) {
                            raise_exception("no-next-method", acc);
                            break;
                        }

                        method = matchmethods(
                                    CAR(acc),
                                    thread,
                                    stack_depth(thread) - argc,
				    CDR(acc));

                        if (method == NULL) {
                            raise_exception("no-next-method", acc);
                            break;
                        }

                        class = GETCLASS(method);

                        if (subclass(class, function_class)) {
                            OBJECT frame = prepare_environment(thread, argc,
                                            (int) NUM(GET(method, FUNC_ARGC)),
                                            method);

                            if (frame != NULL) {
			      OBJECT newenv = NULL;

			      temp_register(&newenv, 1);
			      newenv = frame;
			      newenv = cons(frame, GET(method, FUNC_ENV));
			      push_call(thread, method, CAR(acc), newenv);
			      deregister_root(1);
			    }
                        } else if (subclass(class, prim_class))
                            acc = primitive_call(thread, method, argc);
                        else if (subclass(class, continuation_class))
                            acc = continuation_call(thread, method, argc);
                        else
                            raise_exception("applied-non-method",
                                            newvector(2, 1, CAR(acc), method));

                        break;
                    }

                    case OP_RETURN:
                        pop_call(thread);
                        should_save_state = 0;
                        break;

                    case OP_MAKE_LAMBDA:
                        acc = instantiate_template(
                                IGET(func, buf[pos++]),
                                GET(thread, THR_ENV));
                        continue;

                    default:
                        error("VM Opcode not defined: %x",
                            (word) (unsigned char) buf[pos-1]);
                }

                break;
            }

            if (should_save_state)
                SET(thread, THR_IP, MKNUM(pos));

            continue;
        }

        if (subclass(cls, prim_class)) {
            OBJECT (*cfunc)();
            OBJECT env = GET(thread, THR_ENV);
            int argc;

            pop_call(thread);

            if (ticks > 0 && !interp_critsec)
                ticks--;

            cfunc = getfunc(func);

            argc = abs((int) NUM(GET(func, PRIM_ARGC)));

	    acc = env;	/* To stop env being collected during the execution
			   of the primitive which needs it */

            switch (argc) {
                case 0: acc = cfunc(); break;
                case 1: acc = cfunc(IGET(env, 0)); break;
                case 2: acc = cfunc(IGET(env, 0), IGET(env, 1)); break;
                case 3: acc = cfunc(IGET(env, 0),
                                    IGET(env, 1),
                                    IGET(env, 2));
                        break;
                case 4: acc = cfunc(IGET(env, 0),
                                    IGET(env, 1),
                                    IGET(env, 2),
                                    IGET(env, 3));
                        break;
                case 5: acc = cfunc(IGET(env, 0),
                                    IGET(env, 1),
                                    IGET(env, 2),
                                    IGET(env, 3),
                                    IGET(env, 4));
                        break;
                case 10: acc = cfunc(IGET(env, 0),
                                     IGET(env, 1),
                                     IGET(env, 2),
                                     IGET(env, 3),
                                     IGET(env, 4),
                                     IGET(env, 5),
                                     IGET(env, 6),
                                     IGET(env, 7),
                                     IGET(env, 8),
                                     IGET(env, 9));
                         break;
                case 12: acc = cfunc(IGET(env, 0),
                                     IGET(env, 1),
                                     IGET(env, 2),
                                     IGET(env, 3),
                                     IGET(env, 4),
                                     IGET(env, 5),
                                     IGET(env, 6),
                                     IGET(env, 7),
                                     IGET(env, 8),
                                     IGET(env, 9),
                                     IGET(env, 10),
                                     IGET(env, 11));
                         break;
                case 14: acc = cfunc(IGET(env, 0),
                                     IGET(env, 1),
                                     IGET(env, 2),
                                     IGET(env, 3),
                                     IGET(env, 4),
                                     IGET(env, 5),
                                     IGET(env, 6),
                                     IGET(env, 7),
                                     IGET(env, 8),
                                     IGET(env, 9),
                                     IGET(env, 10),
                                     IGET(env, 11),
                                     IGET(env, 12),
                                     IGET(env, 13));
                        break;
                default:
                    error("I can only handle primitives with argc's of 0, "
                            "1, 2, 3, 4, 5, 10, 12 or 14.\n"
                          "It looks like I was given one with %d!", argc);
            }

            continue;
        }

        raise_exception("invalid-partial-continuation", func);

/*        error("That's strange: the context stack contained an entry which\n"
              "was neither a lambda nor a primitive! The author probably\n"
              "needs to know about this one."); */
    }

    SET(thread, THR_ACC, acc);
    deregister_root(2);
    return false;
}

