/* $Id: string.c,v 1.9 1998/04/05 10:33:48 tonyg Exp $ */

#include "memory.h"
#include "class.h"
#include "string.h"
#include "thread.h"
#include "prim.h"
#include "pair.h"
#include "vector.h"

#include "buffer.h"
#include "scan.h"
#include "parse.h"
#include "compile.h"
#include "interp.h"
#include "symbol.h"
#include "file.h"
#include "filescan.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#ifdef __TURBOC__
#  include <io.h>
#endif

#ifdef unix
#  include <unistd.h>
#endif

OBJECT string_class;

PRIVATE OBJECT load_loop;

void init_string(void) {
  string_class = NULL;
  register_root(&string_class, 1);
  string_class = newclass(object_class, STRING_SIZE, NULL);

  load_loop = NULL;
  register_root(&load_loop, 1);
}

OBJECT newstring(char *val) {
    OBJECT str = NewObject(string_class, 0, strlen(val) + 1);

    strcpy(BIDX(str, 0), val);

    return str;
}

/* Methods */

PRIVATE OBJECT string_printstr(OBJECT str, OBJECT w) {
  BUFFER buf;
  char *s;
  OBJECT result;

  if (w == false)
    return str;

  buf = newbuf(0);

  buf_append(buf, '"');

  for (s = BIDX(str, 0); *s != '\0'; s++)
    switch (*s) {
      case '\r': buf_append(buf, '\\'); buf_append(buf, 'r'); break;
      case '\n': buf_append(buf, '\\'); buf_append(buf, 'n'); break;
      case '\t': buf_append(buf, '\\'); buf_append(buf, 't'); break;
      case '\a': buf_append(buf, '\\'); buf_append(buf, 'a'); break;
      case '\b': buf_append(buf, '\\'); buf_append(buf, 'b'); break;
      case '"': buf_append(buf, '\\'); buf_append(buf, '"'); break;
      case '\\': buf_append(buf, '\\'); buf_append(buf, '\\'); break;
      default: buf_append(buf, *s); break;
    }

  buf_append(buf, '"');
  buf_append(buf, '\0');

  result = newstring(buf->buf);
  killbuf(buf);
  
  return result;
}

typedef struct {
    FILE *f;
    SCANSTATE sc;
    PARSER pa;
} loading_info;

OBJECT string_load(OBJECT filename) {
    FILE *f = fopen(BIDX(filename, 0), "rb");
    OBJECT vec = NULL;
    loading_info *info;

    if (f == NULL)
        return false;

    temp_register(&vec, 1);	/* This is matched by the deregister_root in string_load_loop */

    vec = NewObject(vector_class, 1, sizeof(loading_info));
    ISET(vec, 0, vec);

    info = (loading_info *) BIDX(vec, 0);

    info->f = f;
    info->sc = newscanner(f, file_peek, file_drop);
    info->pa = newparser(scan, info->sc);

    push_call(getcurrthr(), load_loop, NULL, vec);
    return NULL;
}

PRIVATE OBJECT string_load_loop(OBJECT state) {
    loading_info *info = (loading_info *) BIDX(state, 0);
    OBJECT thread;

    if (feof((FILE *) (info->sc->source))) {
        killparser(info->pa);
        killscanner(info->sc);
        fclose(info->f);

        deregister_root(1);	/* This matches the temp_register in string_load */

        return true;
    }

    thread = getcurrthr();
    push_call(thread, load_loop, NULL, state);

    {
        OBJECT thunk = compile(parse(info->pa));

        if (thunk != NULL)
            push_call(thread, thunk, NULL, NULL);
    }

    return NULL;
}

PRIVATE OBJECT string_equal(OBJECT a, OBJECT b) {
    return (!strcmp(BIDX(a, 0), BIDX(b, 0)) ? true : false);
}

PRIVATE OBJECT string_append(OBJECT a, OBJECT b) {
    char *str = getmem(strlen(BIDX(a, 0)) + strlen(BIDX(b, 0)) + 1);
    OBJECT result;

    strcpy(str, BIDX(a, 0));
    strcat(str, BIDX(b, 0));
    result = newstring(str);
    freemem(str);

    return result;
}

PRIVATE OBJECT string_to_sym(OBJECT str) {
    return newsym(BIDX(str, 0));
}

PRIVATE OBJECT string_identity(OBJECT str) {
  return str;
}

PRIVATE OBJECT string_getenv(OBJECT str) {
  char *val = getenv(BIDX(str, 0));

  if (val == NULL)
    return false;
  else
    return newstring(val);
}

PRIVATE OBJECT string_hash_equal(OBJECT obj) {
  return MKNUM(hash_str(BIDX(obj, 0)));	/* Returns a fairly good hash */
					/* Details in symbol.c */
}

PRIVATE OBJECT string_findsub(OBJECT self, OBJECT item) {
  char *pos = strstr(BIDX(self, 0), BIDX(item, 0));

  if (pos == NULL)
    return false;
  else
    return MKNUM(pos - BIDX(self, 0));
}

PRIVATE OBJECT string_cutsub(OBJECT self, OBJECT fromnum, OBJECT lennum) {
  int from = NUM(fromnum);
  int len = NUM(lennum);
  int slen = strlen(BIDX(self, 0));
  char *s;
  OBJECT result;

  if (from < 0)
    from = 0;
  else if (from > slen)
    from = slen;

  if (len < 0)
    len = 0;
  else if ((from + len) > slen)
    len = slen - from;

  s = getmem(len + 1);
  strncpy(s, BIDX(self, 0) + from, len);
  s[len] = '\0';

  result = newstring(s);
  freemem(s);
  return result;
}

PRIVATE OBJECT string_repsub(OBJECT self, OBJECT whatstr, OBJECT withstr) {
  BUFFER buf = newbuf(0);
  char *source, *what, *with, *temp;
  int what_length;
  OBJECT result;
  
  source = BIDX(self, 0);
  what = BIDX(whatstr, 0);
  with = BIDX(withstr, 0);
  what_length = strlen(what);

  while (*source != '\0')
    if (!strncmp(source, what, what_length)) {
      source += what_length;
      temp = with;
      while (*temp != '\0') {
	buf_append(buf, *temp);
	temp++;
      }
    } else {
      buf_append(buf, *source);
      source++;
    }

  buf_append(buf, '\0');

  result = newstring(buf->buf);
  killbuf(buf);

  return result;
}

PRIVATE OBJECT string_toupper(OBJECT self) {
  char *result;
  char *pos;
  OBJECT retval;

  result = getmem(strlen(BIDX(self, 0)) + 1);
  strcpy(result, BIDX(self, 0));

  for (pos = result; *pos != '\0'; pos++)
    if (*pos >= 'a' && *pos <= 'z')
      *pos += 'A' - 'a';

  retval = newstring(result);
  freemem(result);
  return retval;
}

PRIVATE OBJECT string_tolower(OBJECT self) {
  char *result;
  char *pos;
  OBJECT retval;

  result = getmem(strlen(BIDX(self, 0)) + 1);
  strcpy(result, BIDX(self, 0));

  for (pos = result; *pos != '\0'; pos++)
    if (*pos >= 'A' && *pos <= 'Z')
      *pos += 'a' - 'A';

  retval = newstring(result);
  freemem(result);
  return retval;
}

#define AM(n,f,a)   addmeth(n,f,a,cl)

void init_meth_string(void) {
    OBJECT cl = NULL;
    temp_register(&cl, 1);
    cl = cons(string_class, NULL);

    AM("print-string", string_printstr, 2);
    AM("load", string_load, 1);
    AM("equal?", string_equal, 2);
    AM("+", string_append, 2);
    AM("as-symbol", string_to_sym, 1);
    AM("as-string", string_identity, 1);
    AM("getenv", string_getenv, 1);
    AM("hash-for-equal", string_hash_equal, 1);

    AM("find-subseq", string_findsub, 2);
    AM("cut-subseq", string_cutsub, 3);
    AM("replace-subseq", string_repsub, 3);
    AM("to-upper", string_toupper, 1);
    AM("to-lower", string_tolower, 1);

    load_loop = newsym("load-loop");
    load_loop = newprim(1, load_loop, string_load_loop);

    deregister_root(1);
}
