/* $Id: thread.c,v 1.4 1998/04/05 10:33:48 tonyg Exp $ */

#include "memory.h"
#include "class.h"
#include "thread.h"
#include "vector.h"
#include "pair.h"
#include "prim.h"
#include "function.h"
#include "symbol.h"
#include "gf.h"

#include <string.h>

PRIVATE struct {
    OBJECT head, tail;
} tasklist;

PRIVATE OBJECT currthr;

PRIVATE OBJECT exception_dispatcher;

OBJECT thread_class;
OBJECT continuation_class;

void init_thread(void) {
  thread_class = NULL;
  register_root(&thread_class, 1);
  thread_class = newclass(object_class, THR_SIZE, NULL);

  continuation_class = NULL;
  register_root(&continuation_class, 1);
  continuation_class = newclass(object_class, CONT_SIZE, NULL);

  tasklist.head = tasklist.tail = NULL;
  currthr = NULL;
  exception_dispatcher = NULL;

  register_root(&tasklist.head, 1);
  register_root(&currthr, 1);
  register_root(&exception_dispatcher, 1);
}

OBJECT newthread(void) {
    OBJECT thr = NewObject(thread_class, THR_STACK_MAX, 0);

    SET(thr, THR_CONT, NULL);
    SET(thr, THR_IP, MKNUM(0));
    SET(thr, THR_ENV, NULL);
    SET(thr, THR_ACC, NULL);
    SET(thr, THR_FUNC, NULL);
    SET(thr, THR_CURR_GF, NULL);
    SET(thr, THR_STACK_DEPTH, MKNUM(0));
    SET(thr, THR_QUANTUM, MKNUM(50));
    SET(thr, THR_STATUS, MKNUM(0));
    SET(thr, THR_NEXT, NULL);
    SET(thr, THR_PREV, NULL);
    SET(thr, THR_HANDLER_CHAIN, NULL);

    return thr;
}

OBJECT getcurrthr(void) {
    return currthr;
}

OBJECT getcont(OBJECT thread) {
    OBJECT cont = NULL;
    OBJECT vec;
    int i;

    temp_register(&cont, 1);

    cont = NewObject(continuation_class, 0, 0);

    SET(cont, CONT_CONT, GET(thread, THR_CONT));
    vec = newvector(NUM(GET(thread, THR_STACK_DEPTH)), 0);
    SET(cont, CONT_STACK, vec);
    SET(cont, CONT_HANDLER_CHAIN, GET(thread, THR_HANDLER_CHAIN));

    for (i=(int)NUM(GET(thread, THR_STACK_DEPTH))-1; i>=0; i--)
        ISET(vec, i, IGET(thread, i));

    deregister_root(1);

    return cont;
}

void setcont(OBJECT thread, OBJECT cont) {
    int i;
    OBJECT vec = GET(cont, CONT_STACK);

    SET(thread, THR_CONT, GET(cont, CONT_CONT));
    SET(thread, THR_HANDLER_CHAIN, GET(cont, CONT_HANDLER_CHAIN));
    SET(thread, THR_STACK_DEPTH, MKNUM(NUMIDX(vec)));

    for (i=NUMIDX(vec)-1; i>=0; i--)
        ISET(thread, i, IGET(vec, i));
}

void stack_push(OBJECT thread, OBJECT value) {
    word top = (word) NUM(GET(thread, THR_STACK_DEPTH));

    if (top >= THR_STACK_MAX)
        error("Thread stack overflow");

    ISET(thread, top, value);
    SET(thread, THR_STACK_DEPTH, MKNUM(top + 1));
}

OBJECT stack_pop(OBJECT thread) {
    word top = (word) NUM(GET(thread, THR_STACK_DEPTH));

    if (top == 0)
        error("Thread stack underflow");

    top--;
    SET(thread, THR_STACK_DEPTH, MKNUM(top));
    return IGET(thread, top);
}

word stack_depth(OBJECT thread) {
    return (word) NUM(GET(thread, THR_STACK_DEPTH));
}

void apply(OBJECT thread, OBJECT function, OBJECT curr_gf, OBJECT env) {
    if (instance(function, function_class))
        push_call(thread, function, curr_gf,
            cons(env, GET(function, FUNC_ENV)));
    else if (instance(function, prim_class))
        push_call(thread, function, curr_gf,
            env);
    else if (instance(function, continuation_class))
        raise_exception("wrong-argument-count", function);
    else
        raise_exception("applied-non-function", function);
}

void push_call(OBJECT thread, OBJECT function, OBJECT curr_gf, OBJECT env) {
    OBJECT cont = newvector(CTXT_SIZE, 1,
                    GET(thread, THR_CONT),
                    GET(thread, THR_IP),
                    GET(thread, THR_ENV),
                    GET(thread, THR_FUNC),
                    GET(thread, THR_CURR_GF));

    SET(thread, THR_CONT, cont);
    SET(thread, THR_IP, MKNUM(0));
    SET(thread, THR_ENV, env);
    SET(thread, THR_FUNC, function);
    SET(thread, THR_CURR_GF, curr_gf);
}

void pop_call(OBJECT thread) {
    OBJECT cont = GET(thread, THR_CONT);

    SET(thread, THR_CONT, IGET(cont, CTXT_NEXT));
    SET(thread, THR_IP, IGET(cont, CTXT_IP));
    SET(thread, THR_ENV, IGET(cont, CTXT_ENV));
    SET(thread, THR_FUNC, IGET(cont, CTXT_FUNC));
    SET(thread, THR_CURR_GF, IGET(cont, CTXT_CURR_GF));
}

void raise_exception(char *exception, OBJECT args) {
    OBJECT env = NULL;

    temp_register(&env, 1);
    temp_register(&args, 1);

    env = newsym(exception);

    env = newvector(4, 1,
            GET(currthr, THR_HANDLER_CHAIN),
            NULL,
            env,	/* the new symbol, remember? */
            args);

    if (exception_dispatcher == NULL)
        error("No exception dispatcher defined: exception was %s", exception);

    push_call(currthr, exception_dispatcher, NULL, env);

    ISET(env, 1, getcont(currthr));

    deregister_root(2);
}

void unblock_priority(OBJECT thread) {
    if (NUM(GET(thread, THR_STATUS)) == 1)
        block(thread);

    SET(thread, THR_NEXT, tasklist.head);
    SET(thread, THR_PREV, NULL);

    tasklist.head = thread;
    if (tasklist.tail == NULL)
        tasklist.tail = thread;

    SET(thread, THR_QUANTUM, MKNUM(-1));    /* Cannot be preempted. */
    SET(thread, THR_STATUS, MKNUM(1));
}

void unblock(OBJECT thread) {
    if (NUM(GET(thread, THR_STATUS)) == 1)
        block(thread);

    SET(thread, THR_PREV, tasklist.tail);
    SET(thread, THR_NEXT, NULL);

    if (tasklist.head != NULL)
        SET(tasklist.tail, THR_NEXT, thread);
    else
        tasklist.head = thread;

    tasklist.tail = thread;

    SET(thread, THR_STATUS, MKNUM(1));
}

void block(OBJECT thread) {
    OBJECT p, n;

    if (NUM(GET(thread, THR_STATUS)) == 0)
        return;     /* It's already blocked. */

    p = GET(thread, THR_PREV);
    n = GET(thread, THR_NEXT);

    if (p != NULL) SET(p, THR_NEXT, n);
    if (n != NULL) SET(n, THR_PREV, p);

    if (tasklist.head == thread) tasklist.head = n;
    if (tasklist.tail == thread) tasklist.tail = p;

    SET(thread, THR_STATUS, MKNUM(0));
}

OBJECT schedule(void) {
    OBJECT newthr = tasklist.head;

    if (newthr != NULL)
        unblock(newthr);

    currthr = newthr;

    return newthr;
}

void dump_context_on(FILE *f, OBJECT cont) {
    OBJECT ctxt;

    if (cont == NULL) {
        if (currthr == NULL)
            return;

        ctxt = GET(currthr, THR_CONT);
    } else
        ctxt = GET(cont, CONT_CONT);

    while (ctxt != NULL) {
        OBJECT func = IGET(ctxt, CTXT_FUNC);
        char funcname[21];

        if (instance(func, function_class))
            sprintf(funcname, "<function %ld>", NUM(GET(func, FUNC_ARGC)));
        else if (instance(func, prim_class))
            strcpy(funcname, "<primitive-function>");
        else if (func == NULL)
            strcpy(funcname, "<empty-context>");
        else
            strcpy(funcname, "<unknown-callable>");

        {
            int n;
            for (n=strlen(funcname); n<20; n++)
                funcname[n] = ' ';
            funcname[20] = '\0';
        }

        fprintf(f, "    %s at %d: %s (%s)\n",
            funcname,
            (int) NUM(IGET(ctxt, CTXT_IP)),
            func_name(func),
            (IGET(ctxt, CTXT_CURR_GF) == NULL?
                "" :
                BIDX(GET(IGET(ctxt, CTXT_CURR_GF), GF_NAME), 0)));

        if (instance(func, function_class)) {
            func = IGET(ctxt, CTXT_ENV);
            while (func != NULL) {
                OBJECT v = CAR(func);
                int i;

                for (i=0; i<(int)NUMIDX(v); i++)
                    printf("        %d: %p\n", i, IGET(v, i));
                printf("        -----\n");

                func = CDR(func);
            }
        }

        ctxt = IGET(ctxt, CTXT_NEXT);
    }

    fprintf(f, "    at toplevel.\n");
}

/* Methods */

PRIVATE OBJECT thread_currthr(void) {
    return currthr;
}

PRIVATE OBJECT thread_block(OBJECT thread) {
    block(thread);
    return thread;
}

PRIVATE OBJECT thread_unblock(OBJECT thread) {
    unblock(thread);
    return thread;
}

PRIVATE OBJECT thread_unblock_priority(OBJECT thread) {
    unblock_priority(thread);
    return thread;
}

PRIVATE OBJECT the_exception_dispatcher(OBJECT chain, OBJECT cont,
                                        OBJECT ex, OBJECT args) {
  OBJECT R = NULL;

  if (chain == NULL) {
    dump_context_on(stdout, cont);
    error("exception not handled: %s", BIDX(ex, 0));
  }

  temp_register(&R, 1);

  R = newvector(4, 1, CDR(chain), cont, ex, args);
  push_call(currthr, exception_dispatcher, NULL, R);

  R = newvector(3, 1, ex, args, cont);
  apply(currthr, CAR(chain), NULL, R);

  deregister_root(1);
  return NULL;
}

#define AM(n,f,a)   addmeth(n,f,a,cl)

void init_meth_thread(void) {
    OBJECT cl = NULL;
    temp_register(&cl, 1);
    cl = cons(thread_class, NULL);

    AM("block-thread", thread_block, 1);
    AM("unblock-thread", thread_unblock, 1);
    AM("unblock-thread-priority", thread_unblock_priority, 1);

    addprim("current-thread", thread_currthr, 0);

    {
      cl = newsym("Exception Dispatcher");
      exception_dispatcher = newprim(4, cl, the_exception_dispatcher);
    }

    deregister_root(1);
}

