;;@title "GdkEvent binding"
(declare
 (usual-integrations)

 (export gdk-event-type
	 gdk-event-window
	 gdk-event-string
	 gdk-event-area
	 gdk-event-button
	 gdk-event-xy
	 gdk-event-xy-root
	 )

 (foreign-declare #<<EOF

#include <gdk/gdk.h>

EOF
))

;;@ <synopsis>(require 'gdkevent)</synopsis>
;; This extension is automatically included when the gtk extension is
;; <function>require</function>d. It provides accessors for fields in
;; <structname>GdkEvent</structname> boxed structures.
0 ; don't document unwrap

(define-inline (unwrap x)
  (if (GdkEvent? x)
      (g:unbox-GdkEvent x)
      x))

#|
    ((nothing
      map
      unmap
      destroy
      delete)			; any
    ((expose)			; expose
    ((motion-notify)		; motion
    ((button-press
      2button-press
      3button-press
      button-release)		; button
    ((key-press
      key-release)		; key
    ((enter-notify
      leave-notify)		; crossing
    ((focus-change)		; focus_change
    ((configure)		; configure
    ((property-notify)		; property
    ((selection-clear
      selection-request
      selection-notify)		; selection
    ((proximity-in
      proximity-out)		; proximity
    ((drag-enter
      drag-leave
      drag-motion
      drag-status
      drop-start
      drop-finished)		; dnd
    ((client-event)		; client
    ((visibility-notify)	; visibility
    ((no-expose)		; no_expose
|#

(define-macro (gtkevent:eventexpr var rettype . body)
  `((foreign-lambda* ,rettype (((pointer "GdkEvent") ,var))
		     ,@body)
    (unwrap ,var)))

;;@ Retrieves the (symbolic) GdkEventType from a GdkEvent.
(define (gdk-event-type e)
  (number->GdkEventType
   (gtkevent:eventexpr e unsigned-integer "return(e->type);")))

;;@ Retrieves the GdkWindow associated with a GdkEvent.
(define (gdk-event-window e)
  (g:box-GdkWindow
   (gtkevent:eventexpr e c-pointer "return(e->any.window);")))

;;@ Retrieves the string associated with a GdkEvent, or
;;<literal>#f</literal> if there is no associated string. (Currently
;;supports <literal>key-press</literal> and
;;<literal>key-release</literal> events.)
(define (gdk-event-string e)
  (and (memq (gdk-event-type e) '(key-press key-release))
       (gtkevent:eventexpr e c-string "return(e->key.string);")))

;;@ Retrieves the area rectangle of an expose event, or
;;<literal>#f</literal> if the passed-in event is of the wrong type.
(define (gdk-event-area e)
  (and (eq? (gdk-event-type e) 'expose)
       (g:box-GdkRectangle
	(gtkevent:eventexpr e c-pointer "return(&(e->expose.area));"))))

;;@ Retrieves the button number of a button event, or
;;<literal>#f</literal> if the passed-in event is of the wrong type.
(define (gdk-event-button e)
  (and (memq (gdk-event-type e) '(button-press
				  2button-press
				  3button-press
				  button-release))
       (gtkevent:eventexpr e unsigned-integer "return(e->button.button);")))

;;@ Returns two values, the X and Y coordinates associated with a
;;GdkEvent. Returns <literal>(values #f #f)</literal> if there is no
;;associated coordinate pair.
(define (gdk-event-xy e)
  (case (gdk-event-type e)
    ((motion-notify)		; motion
     (values (gtkevent:eventexpr e double "return(e->motion.x);")
	     (gtkevent:eventexpr e double "return(e->motion.y);")))
    ((button-press
      2button-press
      3button-press
      button-release)		; button
     (values (gtkevent:eventexpr e double "return(e->button.x);")
	     (gtkevent:eventexpr e double "return(e->button.y);")))
    ((enter-notify
      leave-notify)		; crossing
     (values (gtkevent:eventexpr e double "return(e->crossing.x);")
	     (gtkevent:eventexpr e double "return(e->crossing.y);")))
    ((configure)		; configure
     (values (gtkevent:eventexpr e short "return(e->configure.x);")
	     (gtkevent:eventexpr e short "return(e->configure.y);")))
    (else (values #f #f))))

;;@ As for <function>gdk-event-xy</function>, except returns
;;coordinates in the root window coordinate system rather than the
;;window-local coordinate system.
(define (gdk-event-xy-root e)
  (case (gdk-event-type e)
    ((motion-notify)		; motion
     (values (gtkevent:eventexpr e double "return(e->motion.x_root);")
	     (gtkevent:eventexpr e double "return(e->motion.y_root);")))
    ((button-press
      2button-press
      3button-press
      button-release)		; button
     (values (gtkevent:eventexpr e double "return(e->button.x_root);")
	     (gtkevent:eventexpr e double "return(e->button.y_root);")))
    ((enter-notify
      leave-notify)		; crossing
     (values (gtkevent:eventexpr e double "return(e->crossing.x_root);")
	     (gtkevent:eventexpr e double "return(e->crossing.y_root);")))
    ((drag-enter
      drag-leave
      drag-motion
      drag-status
      drop-start
      drop-finished)		; dnd
     (values (gtkevent:eventexpr e short "return(e->dnd.x_root);")
	     (gtkevent:eventexpr e short "return(e->dnd.y_root);")))
    (else (values #f #f))))
