;;; Implements a layer on top of database drivers which is (almost)
;;; compatible with "db-util.scm" by Oleg Kiselyov.
;;; See http://okmij.org/ftp/Scheme/#databases
;
; Copyright (c) 2002 Tony Garnock-Jones, based on an original work by
; Oleg Kiselyov.
;
; Permission is hereby granted, free of charge, to any person
; obtaining a copy of this software and associated documentation files
; (the "Software"), to deal in the Software without restriction,
; including without limitation the rights to use, copy, modify, merge,
; publish, distribute, sublicense, and/or sell copies of the Software,
; and to permit persons to whom the Software is furnished to do so,
; subject to the following conditions:
; 
; The above copyright notice and this permission notice shall be
; included in all copies or substantial portions of the Software.
;
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
; EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
; MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
; NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
; BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
; ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
; CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.

;---------------------------------------------------------------------------
; The interface to the database driver, db:
;
; 'error's:
; - are pairs with car #f, cadr a message string, cddr driver-specific details about the error
;
; (db 'close)
; - shuts down the connection, frees all resources
;
; (db 'begin-query stmt)
; - sets up a query for execution
; - returns (#t . an-open-result-set), or an 'error'
;
; (db 'fetch-row result-set)
; - fetches a single row from a result set
; - returns (#t . row) where row is a list of values, or
; - returns #f for no more rows, or
; - returns (#f errormsg . details) (an 'error')
;
; (db 'end-query result-set)
; - releases resources associated with a query
;
; (db 'imperative-stmt stmt)
; - runs an imperative statement
; - returns (#t) or an 'error'

(declare
 (usual-integrations)
 (uses srfi-13)

 (export

  DB:close

  DB:for-each
  DB:imperative-stmt
  DB:for-singleton
  DB:assoc-val
  DB:fetch-column

  DB:quote-string
  DB:trim-whitespace
  DB:quote

  DB:make-sql-stmt-buffer
 )
)

; (DB:close db)
(define (DB:close db)
  (db 'close))

; (DB:check-result result)
(define (DB:check-result result)
  (cond
   ((not (pair? result)) #f)
   ((car result) (cdr result))
   (else
    (apply error "DB:check-result error" (cdr result))
    #f)))

; (DB:for-each proc db [stmt ...])
; (DB:for-each (proc-init . proc) db [stmt ...])
(define (DB:for-each procs db . stmts)
  (let ((proc (if (pair? procs) (cdr procs) procs))
        (proc-init (and (pair? procs) (car procs))))

    (let* ((results (cons 'dummy '()))
	   (final-pair results)

	   (append-result! (lambda (res)
			     (set-cdr! final-pair (cons res '()))
			     (set! final-pair (cdr final-pair))))
	   (get-final-result (lambda ()
			       (cdr results))))

      (let exec1 ((stmts stmts))
	(if (not (null? stmts))
	    (let* ((stmt (car stmts))
		   (result-set (DB:check-result (db 'begin-query stmt))))
	      (let process1 ()
		(let ((row (DB:check-result (db 'fetch-row result-set))))
		  (if row
		      (begin
			(if proc-init (begin (proc-init) (set! proc-init #f)))
			(let ((proc-result (apply proc row)))
			  (cond
			   ; Returning nil from a handler keeps processing without collecting.
			   ((null? proc-result)
			    (process1))
			   ; Returning a true value appends that value to the result list,
			   ; and keeps going
			   (proc-result
			    (append-result! proc-result)
			    (process1))
			   ; Returning a false value causes DB:for-each to return.
			   (else (db 'end-query result-set)))))
		      ; No more rows. Do next stmt.
		      (begin
			(db 'end-query result-set)
			(exec1 (cdr stmts)))))))))

      (get-final-result))))

(define (DB:imperative-stmt db . stmts)
  (for-each (lambda (stmt)
	      (DB:check-result (db 'imperative-stmt stmt)))
	    stmts))

; A variant of the previous function, in a case when we expect
; at most one row in the result
(define (DB:for-singleton proc db stmt1 . stmt-others)
  (let ((have-result? #f))
    (apply DB:for-each
	   (lambda row
	     (if have-result?
		 #f
		 (begin
		   (set! have-result? #t)
		   (apply proc row))))
	   db
	   stmt1
	   stmt-others)))

(define (DB:assoc-val db stmt1 . stmt-others)
  (let ((result (apply DB:for-singleton (lambda lst (car lst)) db stmt1 stmt-others)))
    (if (pair? result)
	(car result)
	#f)))

(define (DB:fetch-column db . stmts)
  (apply DB:for-each (lambda (x . rest) x) db stmts))

;---------------------------------------------------------------------------

(define (DB:quote-string str)
  (list->string
   (cons #\'
	 (string-fold-right (lambda (c acc)
			      (if (memq c '(#\' #\\))
				  (cons* c c acc)
				  (cons c acc)))
			    '(#\')
			    str))))

(define (DB:trim-whitespace datum)
  (cond
   ((string? datum) (string-trim-right datum))
   (else datum)))

(define (DB:quote item)
  (cond
   ((string? item) (DB:quote-string item))
   ((symbol? item) (DB:quote-string (symbol->string item)))
   ((number? item) (number->string item))
   (else item)))

(define (DB:make-sql-stmt-buffer . init-strings)
  (let ((sql-parts (reverse init-strings)))
    (letrec ((accumulate! (lambda (phrases)
			    (for-each 
			     (lambda (phrase)
			       (set! sql-parts (cons phrase sql-parts)))
			     phrases)))
	     (run-stmt (lambda (handler db)
			 (DB:for-each handler
				      db
				      (apply string-append (reverse sql-parts))))))
      (lambda (selector . args)
	(case selector
	  ((accum-sql!) (accumulate! args))
	  ((dump) (display "SQL parts accumulated so far: ")
	          (display (reverse sql-parts))
		  (newline))
	  ((exec) (apply run-stmt args))
	  (else
	   (error "make-sql-stmt-buffer does not understand " selector)))))))
