;;; mysql.scm - Simple mysql glue for Chicken. Use in conjunction with
;;; DB.scm (an implementation of Kiselyov's database interface)
;
; Copyright (c) 2002 Tony Garnock-Jones
;
; Permission is hereby granted, free of charge, to any person
; obtaining a copy of this software and associated documentation files
; (the "Software"), to deal in the Software without restriction,
; including without limitation the rights to use, copy, modify, merge,
; publish, distribute, sublicense, and/or sell copies of the Software,
; and to permit persons to whom the Software is furnished to do so,
; subject to the following conditions:
; 
; The above copyright notice and this permission notice shall be
; included in all copies or substantial portions of the Software.
;
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
; EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
; MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
; NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
; BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
; ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
; CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.

;---------------------------------------------------------------------------

(require 'db)

(declare
 (usual-integrations)

 (export
  DB:mysql-connect
 )

 (foreign-declare #<<EOF
#include <mysql.h>

static char **C_current_row = NULL;

static int C_current_row_fetch(MYSQL_RES *res) {
  C_current_row = mysql_fetch_row(res);
  return (C_current_row != NULL);
}

static char *C_current_field_at(int pos) {
  return C_current_row[pos];
}

EOF
))

(define mysql-init
  (begin
    (define (connect conn . args)
      (let-optionals* args ((host #f)
			    (user #f)
			    (passwd #f)
			    (db #f)
			    (port 0)
			    (unix-socket #f)
			    (clientflag 0))
		      ((foreign-lambda bool "mysql_real_connect"
				       (pointer "MYSQL")
				       c-string
				       c-string
				       c-string
				       c-string
				       integer
				       c-string
				       integer)
		       conn
		       host
		       user
		       passwd
		       db
		       port
		       unix-socket
		       clientflag)))

    (define (encapsulate-result res as-vector)
      (let ((aborted #f))

	(define (release)
	  (if (not aborted)
	      (begin
		(res 'free)
		(set! aborted #t)))
	  #f)

	(lambda args
	  (let ((op (:optional args 'next-result)))
	    (case op

	      ; ((db 'query query) 'next-result)
	      ((next-result)
	       (if aborted
		   #f
		   (let ((row (res (if as-vector 'fetch-row-vector 'fetch-row-list))))
		     (if (not row) (release))
		     row)))

	      ; ((db 'query query) 'free)
	      ((free) (release))

	      (else
	       (error "Unknown cooked mysql-result selector" op)))))))

    (define (run-query db conn query as-vector line-at-a-time)
      (and (db 'raw-query query) ; if this fails, returns #f from run-query.
	   (let ((res (db (if line-at-a-time 'use-result 'store-result))))
	     (cond
	      (res (encapsulate-result res as-vector))
	      ((zero? (db 'field-count)) (lambda args #f)) ; no rows, but none expected
	      (else #f)))))

    (lambda ()
      (let ((conn ((foreign-lambda (pointer "MYSQL") "mysql_init" (pointer "MYSQL")) #f)))
	(define (self op . args)
	  (case op

	    ; (db 'close)
	    ((close)
	     ;(display "Closing Mysql connection ") (display conn) (newline)
	     ((foreign-lambda void "mysql_close" (pointer "MYSQL")) conn))

	    ; (db 'connect [host user passwd db port unix-socket clientflag])
	    ((connect) (apply connect conn args))

	    ; (db 'raw-query query)
	    ((raw-query)
	     (let ((query (car args)))
	       (zero? ((foreign-lambda integer "mysql_real_query"
				       (pointer "MYSQL")
				       c-string
				       unsigned-integer)
		       conn
		       query
		       (string-length query)))))

	    ; (db 'query query)
	    ((query)
	     (let-optionals* (cdr args) ((as-vector #f)
					 (line-at-a-time #f))
			     (run-query self conn (car args)
					as-vector line-at-a-time)))

	    ; (db 'field-count) -- run after query
	    ((field-count)
	     ((foreign-lambda unsigned-integer "mysql_field_count" (pointer "MYSQL")) conn))

	    ; (db 'store-result) -- run after query
	    ((store-result)
	     (mysql-make-result
	      ((foreign-lambda (pointer "MYSQL_RES") "mysql_store_result" (pointer "MYSQL"))
	       conn)))

	    ; (db 'use-result) -- run after query
	    ((use-result)
	     (mysql-make-result
	      ((foreign-lambda (pointer "MYSQL_RES") "mysql_use_result" (pointer "MYSQL"))
	       conn)))

	    (else
	     (error "Unknown mysql db selector" op))))

	self))))

(define mysql-make-result
  (begin

    (define fetch-row (foreign-lambda bool "C_current_row_fetch" (pointer "MYSQL_RES")))
    (define field-at (foreign-lambda c-string "C_current_field_at" integer))

    (lambda (res)
      (and res
	   (begin
	     (define (self op . args)
	       (case op

					; (res 'free)
		 ((free)
		  ;(display "Freeing result ") (display res) (newline)
		  ((foreign-lambda void "mysql_free_result" (pointer "MYSQL_RES"))
		   res))

					; (res 'num-rows)
		 ((num-rows)
		  ((foreign-lambda unsigned-long "mysql_num_rows" (pointer "MYSQL_RES"))
		   res))

					; (res 'num-fields)
		 ((num-fields)
		  ((foreign-lambda unsigned-integer "mysql_num_fields"
				   (pointer "MYSQL_RES"))
		   res))

					; (res 'fetch-row-list)
		 ((fetch-row-list)
		  (and (fetch-row res)
		       (let ((field-count (self 'num-fields)))
			 (let rows-from ((x 0))
			   (if (= x field-count)
			       '()
			       (cons (field-at x) (rows-from (+ x 1))))))))

					; (res 'fetch-row-vector)
		 ((fetch-row-vector)
		  (and (fetch-row res)
		       (let* ((field-count (mysql-num-fields res))
			      (result (make-vector field-count #f)))
			 (let rows-from ((x 0))
			   (if (= x field-count)
			       result
			       (begin
				 (vector-set! result x (field-at x))
				 (rows-from (+ x 1))))))))

		 (else
		  (error "Unknown raw mysql-result selector" op))))

	     self)))))

; (DB:mysql-connect [host [user [passwd [db [port [unix-socket [clientflag]]]]]]])

(define (DB:mysql-connect . args)
  (let ((handle (mysql-init)))
    (if (not (apply handle 'connect args))
	#f
	(lambda (selector . args)
	  (case selector

	    ((close)
	     (handle 'close))

	    ((begin-query)
	     (let* ((stmt (car args))
		    (rs (handle 'query stmt)))
	       (if rs
		   (cons #t rs)
		   (list #f "mysql begin-query failed" stmt))))

	    ((fetch-row)
	     (let* ((rs (car args))
		    (row (and rs (rs))))
	       (and row (cons #t row))))

	    ((end-query)
	     (let ((rs (car args)))
	       (and rs (rs 'free))))

	    ((imperative-stmt)
	     (let* ((stmt (car args))
		    (rs (handle 'query stmt)))
	       (if rs
		   (begin
		     (rs 'free)
		     (cons #t '()))
		   (list #f "mysql imperative-stmt failed" stmt))))

	    (else
	     (error "DB:mysql-connect handler does not understand" selector args)))))))
