;;; socket.scm - Simple socket glue for Chicken
;
; Copyright (c) 2002 Tony Garnock-Jones
;
; Permission is hereby granted, free of charge, to any person
; obtaining a copy of this software and associated documentation files
; (the "Software"), to deal in the Software without restriction,
; including without limitation the rights to use, copy, modify, merge,
; publish, distribute, sublicense, and/or sell copies of the Software,
; and to permit persons to whom the Software is furnished to do so,
; subject to the following conditions:
; 
; The above copyright notice and this permission notice shall be
; included in all copies or substantial portions of the Software.
;
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
; EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
; MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
; NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
; BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
; ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
; CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
; SOFTWARE.

; ---------------------------------------------------------------------------
; Things that would be good to see in here:
;   - some better error checking (!)
;   - UDP support
;   - use file-read/file-write where posix is definitely available maybe...?
;
; Please send any comments, bug reports, suggestions, or patches to
; comp.lang.scheme, the Chicken mailinglist, or directly to me (Google
; finds me efficiently), as the mood takes you...

; ' <- make emacs happy

(declare
 (uses srfi-4 extras lolevel posix)

 (fixnum)

 ;; %%% export declaration should go here

 (usual-integrations)
 (foreign-declare #<<EOF

#include <unistd.h>
#include <fcntl.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>

#include <sys/un.h>
#include <netinet/in.h>
#include <netinet/tcp.h>

#define C_getprotobyname(name) ({struct protoent *e = getprotobyname(name); \
				 e ? e->p_proto : -1; \
			       })

/* Taken directly from unistd.scm */
#define C_read(fd, b, n)    C_fix(read(C_unfix(fd), C_data_pointer(b), C_unfix(n)))
#define C_write(fd, b, n)   C_fix(write(C_unfix(fd), C_data_pointer(b), C_unfix(n)))

#define C_sa_family(bv)		C_fix(((struct sockaddr *) C_data_pointer(bv))->sa_family)
#define C_sa_family_set(bv,f)	(((struct sockaddr *) C_data_pointer(bv))->sa_family = C_unfix(f),\
				 C_SCHEME_UNDEFINED)

#define C_sin_port(bv)		C_fix(ntohs(((struct sockaddr_in *) C_data_pointer(bv))->sin_port))
#define C_sin_port_set(bv,p)	(((struct sockaddr_in *) C_data_pointer(bv))->sin_port = htons((unsigned short) C_num_to_int(p)), C_SCHEME_UNDEFINED)

#define C_get_sin_addr(bv,n)						\
  (memcpy(C_data_pointer(n),						\
	  &((struct sockaddr_in *) C_data_pointer(bv))->sin_addr,	\
	  sizeof(((struct sockaddr_in *) 0)->sin_addr)),		\
   n)

#define C_set_sin_addr(bv,n)						\
  (memcpy(&((struct sockaddr_in *) C_data_pointer(bv))->sin_addr,	\
	  C_data_pointer(n),						\
	  sizeof(((struct sockaddr_in *) 0)->sin_addr)),		\
   bv)

#define C_sun_path_set(bv,p)	(memcpy(((struct sockaddr_un *) C_data_pointer(bv))->sun_path,	\
					C_c_string(p),						\
					C_header_size(p)),					\
				 ((struct sockaddr_un *) C_data_pointer(bv))			\
				    -> sun_path[C_header_size(p)] = '\0',			\
				 C_SCHEME_UNDEFINED)

static struct hostent *C_hostent;

#define C_hostent_alias_isnull(x)	C_mk_bool(C_hostent->h_aliases[C_unfix(x)] == NULL)
#define C_hostent_addrtype()		C_fix(C_hostent->h_addrtype)
#define C_hostent_length()		C_fix(C_hostent->h_length)
#define C_hostent_addr_isnull(x)	C_mk_bool(C_hostent->h_addr_list[C_unfix(x)] == NULL)
#define C_get_hostent_addr(x,bv)	(memcpy(C_data_pointer(bv),			\
						C_hostent->h_addr_list[C_unfix(x)],	\
						C_hostent->h_length),			\
					 bv)

static unsigned char C_sock_temp_bv[4096];
static socklen_t C_sock_temp_bv_len;

#define C_sock_int_from_temp()		C_fix(((int *) &C_sock_temp_bv[0])[0])
#define C_sock_copy_temp(dest)		(memcpy(C_data_pointer(dest),	\
						C_sock_temp_bv,		\
						C_sock_temp_bv_len),	\
					 dest)
#define C_sock_get_temp_len()		C_fix(C_sock_temp_bv_len)

EOF
))

;---------------------------------------------------------------------------
; Low-level socket interface.

(define-foreign-variable _pf_unix int "PF_UNIX")
(define-foreign-variable _pf_inet int "PF_INET")
;(define-foreign-variable _pf_inet6 int "PF_INET6")

(define pf/unix _pf_unix)
(define pf/inet _pf_inet)
;(define pf/inet6 _pf_inet6)

(define-foreign-variable _sock_stream int "SOCK_STREAM")
(define-foreign-variable _sock_dgram int "SOCK_DGRAM")

(define sock/stream _sock_stream)
(define sock/dgram _sock_dgram)

(define socket:getprotobyname (foreign-lambda int "C_getprotobyname" c-string))

(define socket:htons (foreign-lambda integer "htons" integer))
(define socket:htonl (foreign-lambda integer "htonl" integer))
(define socket:ntohs (foreign-lambda integer "ntohs" integer))
(define socket:ntohl (foreign-lambda integer "ntohl" integer))

;; struct sockaddr

(define-record sockaddr bytes)

(define (sockaddr-family sa)
  (assert (sockaddr? sa))
  (##core#inline "C_sa_family" (sockaddr-bytes sa)))

;; printing sockaddr structs

(define *socket:family-sockaddr-printers* '())

(define (socket:register-sockaddr-printer! family printer)
  (let ((cell (assq family *socket:family-sockaddr-printers*)))
    (if cell
	(error "Redefinition of sockaddr-printer"
	       family)
	(set! *socket:family-sockaddr-printers*
	      (cons (cons family printer)
		    *socket:family-sockaddr-printers*)))))

(define-record-printer (sockaddr sa out)
  (let ((cell (assq (sockaddr-family sa) *socket:family-sockaddr-printers*)))
    (if cell
	((cdr cell) sa out)
	(fprintf out "#<sockaddr:~a>"
		 (sockaddr-family sa)))))

;; struct sockaddr_in

(define-foreign-variable _sizeof/sockaddr_in int "sizeof(struct sockaddr_in)")
(define-constant sizeof/in_addr 4)

(assert (eq? sizeof/in_addr
	     ((foreign-lambda* int () "return(sizeof(struct in_addr));"))))

(define (make-sockaddr_in port inaddr)
  (if (not inaddr)
      (set! inaddr (byte-vector 0 0 0 0)))

  (assert (number? port))
  (assert (byte-vector? inaddr))

  (let* ((bv (make-byte-vector _sizeof/sockaddr_in 0))
	 (sa (make-sockaddr bv)))
    (##core#inline "C_sa_family_set" bv _pf_inet)
    (##core#inline "C_sin_port_set" bv port)
    (##core#inline "C_set_sin_addr" bv inaddr)
    sa))

(define (sockaddr_in-port sa)
  (assert (sockaddr? sa))
  (##core#inline "C_sin_port" (sockaddr-bytes sa)))

(define (sockaddr_in-port-set! sa p)
  (assert (sockaddr? sa))
  (assert (number? p))
  (##core#inline "C_sin_port_set" (sockaddr-bytes sa) p))

(define (sockaddr_in-addr sa)
  (assert (sockaddr? sa))
  (let ((result (make-byte-vector sizeof/in_addr)))
    (##core#inline "C_get_sin_addr" (sockaddr-bytes sa) result)
    result))

(socket:register-sockaddr-printer! pf/inet
				   (lambda (sa out)
				     (let ((bv (sockaddr_in-addr sa)))
				       (fprintf out "#<sockaddr_in:~a.~a.~a.~a:~a>"
						(byte-vector-ref bv 0)
						(byte-vector-ref bv 1)
						(byte-vector-ref bv 2)
						(byte-vector-ref bv 3)
						(sockaddr_in-port sa)))))

;; struct sockaddr_un

(define-foreign-variable _sizeof/sockaddr_un int "sizeof(struct sockaddr_un)")

(define (make-sockaddr_un path)
  (assert (string? path))
  (let* ((bv (make-byte-vector _sizeof/sockaddr_un 0))
	 (sa (make-sockaddr bv)))
    (##core#inline "C_sa_family_set" bv _pf_unix)
    (##core#inline "C_sun_path_set" bv path)
    sa))

(define (sockaddr_un-path sa)
  (assert (sockaddr? sa))
  ((foreign-lambda* c-string ((byte-vector bv))
		    "return(((struct sockaddr_un *) bv)->sun_path);")
   (sockaddr-bytes sa)))

(define (sockaddr_un-path-set! sa path)
  (assert (sockaddr? sa))
  (##core#inline "C_sun_path_set" (sockaddr-bytes sa) path))

(socket:register-sockaddr-printer! pf/unix
				   (lambda (sa out)
				     (fprintf out "#<sockaddr_un:~a>"
					      (sockaddr_un-path sa))))

;; struct hostent

(define-record hostent name aliases addrtype length addr_list)

(define-record-printer (hostent h out)
  (fprintf out "#<hostent ~a ~a>"
	   (hostent-name h)
	   (map byte-vector->list (hostent-addr_list h))))

(define (hostent-addr h)
  (first (hostent-addr_list h)))

;;--

(define-inline (%result-check name result)
  (if (= -1 result)
      (##sys#update-errno))
  (zero? result))

(define socket:socket
  (lambda (family kind . proto)
    (if (null? proto)
	(set! proto 0)
	(set! proto (car proto)))
    (let retry ((retried #f))
      (let ((fd ((foreign-lambda int "socket" int int int) family kind proto)))
	(if (= -1 fd)
	    (if retried
		(begin
		  (##sys#update-errno)
		  #f)
		(begin
		  (gc #t)	; collects and forces finalizers
		  (retry #t)))
	    fd)))))

(define socket:shutdown
  (let ((%shutdown (foreign-lambda int "shutdown" int int)))
    (lambda (fd how)
      (%result-check "shutdown"
		     (%shutdown fd
				(if (number? how) how (case how
							((read) 0)
							((write) 1)
							((both) 2)
							(else 2))))))))

(define (socket:close fd)
  (%result-check "close"
		 ((foreign-lambda int "close" int) fd)))

(define socket:read ; adapted from posix file-read
  (let ((make-string make-string))
    (lambda (fd size . rest)
      (let* ((buf (:optional rest (make-string size)))
	     (result (##core#inline "C_read" fd buf size)))
	(if (= -1 result)
	    (##sys#update-errno))
	(values buf result)))))

(define socket:write ; adapted from posix file-write
  (let ((make-string make-string))
    (lambda (fd buf . rest)
      (let* ((size (:optional rest (##sys#size buf)))
	     (result (##core#inline "C_write" fd buf size)))
	(if (= -1 result)
	    (begin (##sys#update-errno) #f)
	    result)))))

(define (%extract-hostent)
  (let ((h (make-hostent ((foreign-lambda* c-string () "return(C_hostent->h_name);"))
			 '()
			 (##core#inline "C_hostent_addrtype")
			 (##core#inline "C_hostent_length")
			 '())))
    (let loop ((i 0)
	       (res '()))
      (if (##core#inline "C_hostent_alias_isnull" i)
	  (hostent-aliases-set! h (reverse! res))
	  (loop (fx+ i 1)
		(cons ((foreign-lambda* c-string ((int i))
					"return(C_hostent->h_aliases[i]);")
		       i)
		      res))))
    (let ((len (hostent-length h)))
      (let loop ((i 0)
		 (res '()))
	(if (##core#inline "C_hostent_addr_isnull" i)
	    (hostent-addr_list-set! h (reverse! res))
	    (loop (fx+ i 1)
		  (cons (##core#inline "C_get_hostent_addr" i (make-byte-vector len))
			res)))))
    h))

(define (socket:gethostbyname hostname)
  (and ((foreign-lambda* bool ((c-string hostname))
			 "C_hostent = gethostbyname(hostname);"
			 "return(C_hostent != NULL);")
	hostname)
       (%extract-hostent)))

(define (socket:gethostbyaddr addr family)
  (assert (byte-vector? addr))
  (assert (number? family))
  (and ((foreign-lambda* bool ((byte-vector addr)
			       (int len)
			       (int family))
			 "C_hostent = gethostbyaddr(addr, len, family);"
			 "return(C_hostent != NULL);")
	addr
	(byte-vector-length addr)
	family)
       (%extract-hostent)))

(define (socket:bind fd sa)
  (assert (sockaddr? sa))
  (let* ((bv (sockaddr-bytes sa)))
    (%result-check "bind"
		   ((foreign-lambda* int ((int fd) (byte-vector sa) (int salen))
				     "return(bind(fd, (struct sockaddr *) sa, salen));")
		    fd
		    bv
		    (byte-vector-length bv)))))

(define (socket:setsockopt fd level optname val)
  (assert (fixnum? fd))
  (assert (fixnum? level))
  (assert (fixnum? optname))
  (cond
   ((number? val)
    (%result-check "setsockopt(int)"
		   ((foreign-lambda* int ((int fd) (int level) (int optname)
					  (int val))
				     "int val_copy = val;"
				     "return(setsockopt(fd, level, optname,"
				     "       &val_copy, sizeof(int)));")
		    fd level optname
		    val)))
   ((byte-vector? val)
    (%result-check "setsockopt(bv)"
		   ((foreign-lambda* int ((int fd) (int level) (int optname)
					  (byte-vector bv) (int len))
				     "return(setsockopt(fd, level, optname,"
				     "       bv, len));")
		    fd level optname
		    val (byte-vector-length val))))
   (else
    (error "socket:setsockopt: expects number or byte-vector"
	   val))))

(define (%getsockopt fd level optname)
  (%result-check "getsockopt"
		 ((foreign-lambda* int ((int fd) (int level) (int optname))
				   "C_sock_temp_bv_len = sizeof(C_sock_temp_bv);"
				   "return(getsockopt(fd, level, optname,"
				   "                  &C_sock_temp_bv[0],"
				   "                  &C_sock_temp_bv_len));")
		  fd level optname)))

(define (socket:getsockopt fd level optname)
  (critical-section
   (and (%getsockopt fd level optname)
	(##core#inline "C_sock_int_from_temp"))))

(define (socket:getsockopt-byte-vector fd level optname)
  (critical-section
   (and (%getsockopt fd level optname)
	(let ((bv (make-byte-vector (##core#inline "C_sock_get_temp_len"))))
	  (##core#inline "C_sock_copy_temp" bv)
	  bv))))

(define-foreign-variable _sol_socket int "SOL_SOCKET")
(define-foreign-variable _so_reuseaddr int "SO_REUSEADDR")

(define-foreign-variable _ipproto_tcp int "IPPROTO_TCP")
(define-foreign-variable _tcp_nodelay int "TCP_NODELAY")

(define (socket:reuse-addr! fd . val)
  (socket:setsockopt fd _sol_socket _so_reuseaddr (if (:optional val #t) 1 0)))

(define (socket:tcp-no-delay! fd . val)
  (socket:setsockopt fd _ipproto_tcp _tcp_nodelay (if (:optional val #t) 1 0)))

(define socket:set-nonblocking!
  (let ((%setnb (foreign-lambda* int ((int fd) (int val))
				 "if (val) "
				 "  return(fcntl(fd, F_SETFL, fcntl(fd, F_GETFL, 0)"
				 "                            | O_NONBLOCK));"
				 "else"
				 "  return(fcntl(fd, F_SETFL, fcntl(fd, F_GETFL, 0)"
				 "                            & ~O_NONBLOCK));")))
    (lambda (fd . val)
      (%result-check "setnonblocking"
		     (%setnb fd (if (:optional val #t) 1 0))))))

(define (socket:listen fd qlen)
  (%result-check "listen"
		 ((foreign-lambda int "listen" int int) fd qlen)))

(define (socket:accept servfd suspender)
  (gc #t) ; forces finalizers - may free up some sockets...
  (let-values (((fd bv)
		(critical-section
		 (let ((result ((foreign-lambda* int ((int servfd))
						 "C_sock_temp_bv_len = sizeof(C_sock_temp_bv);"
						 "return(accept(servfd,"
						 "       (struct sockaddr *) &C_sock_temp_bv,"
						 "       &C_sock_temp_bv_len));")
				servfd)))
		   (if (= -1 result)
		       (begin
			 (##sys#update-errno)
			 (values #f #f))
		       (let ((bv (make-byte-vector (##core#inline "C_sock_get_temp_len"))))
			 (##core#inline "C_sock_copy_temp" bv)
			 (values result bv)))))))
    (if fd
	(list fd (make-sockaddr bv))
	(begin
	  (if (and suspender
		   (equal? (errno) errno/again))
	      (begin
		(suspender 'accepting servfd)
		(socket:accept servfd suspender))
	      #f)))))

(define socket:connect
  (let ((%connect (foreign-lambda* int ((int fd) (byte-vector bv) (int len))
				   "return(connect(fd, (struct sockaddr *) bv, len));")))
    (lambda (fd sa)
      (assert (sockaddr? sa))
      (let ((bv (sockaddr-bytes sa)))
	(%result-check "connect"
		       (%connect fd bv (byte-vector-length bv)))))))

(define-foreign-variable _einprogress int "EINPROGRESS")
(define errno/inprogress _einprogress)

(define socket:connect-nonblocking
  (let ((%geterr (foreign-lambda* int ((int fd))
				  "int sockerr = 0;"
				  "socklen_t sockerrlen = sizeof(sockerr);"
				  "if (getsockopt(fd, SOL_SOCKET, SO_ERROR, "
				  "               &sockerr, &sockerrlen) == -1) {"
				  "  return(-1);"
				  "}"
				  "if (sockerr == EINPROGRESS) return(1);"
				  "if (sockerr == 0) return(0);"
				  "return(-1);")))
    (lambda (fd raw-connect-result suspender)
      (cond

       ; Original connect(2) succeeded:
       (raw-connect-result #t)

       ; Original connect(2) failed, but we have a block handler and
       ; it wanted us to wait a bit:
       ((and suspender
	     (equal? (errno) errno/inprogress))
	(let retry ()
	  (suspender 'connecting fd)
	  (case (%geterr fd)
	    ((0) ; connected ok
	     #t)
	    ((1) ; try again
	     (retry))
	    (else ; error
	     #f))))

       ; Original connect(2) failed in some other way, or we don't
       ; care how it failed:
       (else #f)))))

;---------------------------------------------------------------------------
; More schemeish socket interface.

(define socket:*tcp-proto-number* (socket:getprotobyname "tcp"))

(define (name->sockaddr_in port name)
  (cond
   ((not name) (make-sockaddr_in port name))
   ((byte-vector? name) (make-sockaddr_in port name))
   ((u8vector? name) (make-sockaddr_in port (u8vector->byte-vector name)))
   ((vector? name) (make-sockaddr_in (list->byte-vector port (vector->list name))))
   ((string? name)
    (let ((h (socket:gethostbyname name)))
      (and h
	   (make-sockaddr_in port (hostent-addr h)))))
   (else (error "name->sockaddr_in"
		'(byte-vector u8vector vector string)
		name))))

(define (SOCKET:ports-for-fd fd suspender)
  (let* ((refcount 2)
	 (make-closer (lambda (how)
			(let ((already-run? #f))
			  (lambda ()
			    (if already-run?
				#t
				(begin
				  (set! already-run? #t)
				  (set! refcount (- refcount 1))
				  (socket:shutdown fd how)
				  (if (zero? refcount) (socket:close fd)))))))))
;    (cout "Opening ports on " fd "\n")
    (list

     ; Result is a list. First element is the input port.
     (let* ((bufsize 1024)
	    (buf (make-string 1024))
	    (pos 0)
	    (len 0))
       (let ((input-port (make-input-port
			  (lambda ()
			    (call-with-current-continuation
			     (lambda (return)
			       (do ()
				   ((< pos len)
				    (let ((result (string-ref buf pos)))
				      (set! pos (+ pos 1))
				      (return result)))
				 (let-values (((retbuf n) (socket:read fd bufsize buf)))
				   (cond
				    ((positive? n)
				     (set! pos 0)
				     (set! len n))
				    ((zero? n)
				     (return (end-of-file)))
				    ((and suspender
					  (equal? (errno) errno/again))
				     (suspender 'reading fd))
				    (else
				     (##sys#signal-hook #:file-error
							"cannot read from socket"
							fd
							(errno)))))))))
			  (lambda () #f) ; %%% this is char-ready? - fixme
			  (make-closer 'read))))
	 (set-finalizer! input-port close-input-port)
	 (set-port-name! input-port (with-output-to-string
				      (lambda ()
					(display "socket:")
					(display fd))))
	 input-port))

     ; Second element is the output port.
     (let ((output-port (make-output-port
			 (lambda (str)
			   (let retry ()
			     (let ((result (socket:write fd str)))
			       (if (negative? result)
				   (if (and suspender
					    (equal? (errno) errno/again))
				       (begin
					 (suspender 'writing fd)
					 (retry))
				       (##sys#signal-hook #:file-error
							  "cannot write to socket"
							  fd
							  (errno)))
				   result))))
			 (make-closer 'write))))
       (set-finalizer! output-port close-output-port)
       (set-port-name! output-port (with-output-to-string
				     (lambda ()
				       (display "socket:")
				       (display fd))))
       output-port))))

(define (open-tcp-socket hostname portnumber suspender . args)
  (call-with-current-continuation
   (lambda (return)
     (let-optionals args
	 ((local-bind-hostname #f)
	  (local-bind-portnumber 0))
       (let* ((local-bind? (or local-bind-hostname (not (zero? local-bind-portnumber))))
	      (local-bind-addr (name->sockaddr_in local-bind-portnumber
						  local-bind-hostname))
	      (hostaddr (name->sockaddr_in portnumber hostname))
	      (fd (socket:socket pf/inet sock/stream socket:*tcp-proto-number*)))

	 (unless fd
	   (return #f))

	 (when suspender
	   (socket:set-nonblocking! fd #t))

	 (when local-bind?
	   (unless (socket:bind fd local-bind-addr)
	     (return #f)))

	 (unless (socket:connect-nonblocking fd
					     (socket:connect fd hostaddr)
					     suspender)
	   (return #f))

	 (socket:ports-for-fd fd suspender))))))

(define (open-unix-socket pathname suspender)
  (call-with-current-continuation
   (lambda (return)
     (let ((fd (socket:socket pf/unix sock/stream 0)))
       (unless fd
	 (return #f))

       (when suspender
	 (SOCKET:set-nonblocking! fd #t))

       (unless (socket:connect-nonblocking fd
					   (socket:connect fd (make-sockaddr_un pathname))
					   suspender)
	 (return #f))

       (socket:ports-for-fd fd suspender)))))

(define (make-server-socket addr . args)
  (call-with-current-continuation
   (lambda (return)
     (let ((proto (get-keyword #:protocol args (constantly 0)))
	   (suspender (get-keyword #:suspender args))
	   (reuse? (get-keyword #:reuse? args))
	   (max-allow-wait (get-keyword #:max-allow-wait args (constantly 4))))
       (let ((fd (socket:socket (sockaddr-family addr) sock/stream proto)))

	 (unless fd
	   (return #f))

	 (when suspender
	   (SOCKET:set-nonblocking! fd #t))

	 (when reuse?
	   (unless (socket:reuse-addr! fd)
	     (return #f)))

	 (unless (and (socket:bind fd addr)
		      (socket:listen fd max-allow-wait))
	   (return #f))

	 fd)))))

(define (socket:accept-ports fd suspender)
  (and fd
       (let ((conn (socket:accept fd suspender)))
	 (and conn
	      (let ((newfd (first conn))
		    (newaddr (second conn)))
		(when suspender
		  (socket:set-nonblocking! newfd #t))
		(append (socket:ports-for-fd newfd suspender) (list newaddr)))))))

(define (run-socket-server addr connection-handler . args)
  (let ((suspender (get-keyword #:suspender args)))
    (let ((fd (apply make-server-socket addr args)))
      (and fd
	   (let accept-another ()
	     (let ((accepted (socket:accept-ports fd suspender)))
	       (and accepted
		    (begin
		      (apply connection-handler accepted)
		      (accept-another)))))))))

(define (run-tcp-server hostname portnumber connection-handler . args)
  (apply run-socket-server
	 (name->sockaddr_in portnumber hostname)
	 connection-handler
	 args))

(define (run-unix-server pathname connection-handler . args)
  (apply run-socket-server
	 (make-sockaddr_un pathname)
	 connection-handler
	 args))
