;;@title "extract-doc: Scheme comments to SXML"
(declare
 (uses script-utils srfi-1 srfi-13 srfi-18 regex extras)
)

;;@sxml
;; (para (cmdsynopsis
;;        (command "extract-doc")
;;        (arg (@ (rep "repeat")
;; 	       (choice "plain"))
;; 	    (replaceable "filename")))
;;       "Extracts documentation from Scheme files and writes out
;; "(acronym "SXML")" DocBook entries to stdout. SRFI-10 external
;; representations of comment blocks are used as an intermediate
;; processing step.")

;;@section "Internal datatypes and procedures"

;;@ Holds a comment-block between parsing and output.
(define-record comment-block command lines code)

(define-reader-ctor 'comment-block
  (lambda (command . lines)
    (make-comment-block command lines #f)))

(define-record-printer (comment-block b out)
  (with-output-to-port out
    (lambda ()
      (print* "#<comment-block "
	      (comment-block-command b) " "
	      (comment-block-lines b) " "
	      (comment-block-code b) ">"))))

;;@ Convert an arbitrary scheme object to a string.
(define (->string x)
  (string-trim-right
   (with-output-to-string
     (lambda ()
       (pretty-print x)))))

(define (reverse-car p)
  (cons (reverse (car p)) (cdr p)))

;;@ Splits a list of strings into a list of paragraphs, representing
;;each paragraph as a list of strings. Empty (zero-length) strings are
;;interpreted as paragraph separators in the input list.
(define (split-paragraphs lines)
  (reverse!
   (remove null?
	   (reverse-car
	    (fold (lambda (line acc)
		    (if (zero? (string-length line))
			(cons '() (reverse-car acc))
			(cons (cons line (car acc)) (cdr acc))))
		  (list '())
		  lines)))))

;;@ Emits a block of comment text to the output SXML document, after
;;splitting it into paragraphs using
;;<function>split-paragraphs</function> and formatting it
;;appropriately.
(define (gen-body-text doc lines)
  (let ((paras (split-paragraphs lines)))
    (if (null? paras)
	(doc 'append-item! `(para))
	(for-each (lambda (para)
		    (doc
		     'append-item!
		     `(para (*literal*
			     ,@(map (lambda (x)
				      (string-append x "\n"))
				    para)))))
		  paras))))

;;@ A version of gensym that generates (reasonably) unique symbols not
;;only within an instance of the scheme system, but also between runs,
;;by incorporating the current time in the symbol.
(define newsym
  (let* ((u (inexact->exact (truncate (* 1000 (remainder (time->seconds (current-time)) 1000)))))
	 (s (string-append "docscm" (number->string u) "-")))
    (lambda ()
      (gensym s))))

;;@ Formats a block of documentation for a variable as DocBook markup.
(define (gen-variable-def doc lines varname initlist)
  (doc 'append-item! `(indexterm (@ (id ,(newsym)))
				 (primary "Variables") (secondary (varname ,varname))))
  (doc
   'append-item!
   `(fieldsynopsis (varname ,varname)
		   ,@(if (null? initlist)
			 '()
			 `((initializer ,(->string (car initlist)))))))
  (gen-body-text doc lines))

;;@ Formats a block of documentation for a function as DocBook markup.
(define (gen-function-def doc lines fnname arglist)
  (doc 'append-item! `(indexterm (@ (id ,(newsym)))
				 (primary "Functions") (secondary (function ,fnname))))
  (doc
   'append-item!
   `(methodsynopsis (methodname ,fnname)
		    ,@(if (null? arglist)
			  `((void ""))
			  (let loop ((arglist arglist))
			    (cond
			     ((null? arglist) '())
			     ((pair? arglist)
			      (cons `(methodparam (parameter ,(->string (car arglist))))
				    (loop (cdr arglist))))
			     (else
			      `((methodparam (@ (rep "repeat"))
					     (parameter ,(->string arglist))))))))))
  (gen-body-text doc lines))

;;@ Formats a block of documentation for a macro as DocBook markup.
(define (gen-macro-def doc lines macroname arglist)
  (doc 'append-item! `(indexterm (@ (id ,(newsym)))
				 (primary "Macros") (secondary (function ,macroname))))
  (doc
   'append-item!
   `(methodsynopsis (@ (role "macro"))
		    (methodname ,macroname)
		    ,@(let loop ((arglist arglist))
			(cond
			 ((null? arglist) '())
			 ((pair? arglist)
			  (cons `(methodparam (parameter ,(car arglist)))
				(loop (cdr arglist))))
			 (else
			  `((methodparam (@ (rep "repeat"))
					 (parameter ,arglist))))))))
  (gen-body-text doc lines))

;;@ Appends raw SXML text directly to the output SXML document. Used
;;by the <command>@sxml</command> <command>docscm</command> command.
(define (gen-sxml doc lines)
  (for-each (lambda (x)
	      (doc 'append-item! x))
	    (with-input-from-string
		(string-join lines "\n")
	      read-file)))

;;@ Recurses into an expression, trying (via a fairly sloppy
;;algorithm) to figure out if it returns a procedure or not.
(define (dig-for-lambda-value body)
  (cond
   ((and (pair? body)
	 (eq? (car body) 'lambda))
    (cadr body))
   ((list? body)
    (dig-for-lambda-value (last body)))
   (else #f)))

(define (make-define-like-handler fnlike-generator)
  (lambda (doc lines code)
    (let ((name (second code)))
      (if (pair? (second code))
	  (fnlike-generator doc lines (car name) (cdr name))
	  (cond
	   ((dig-for-lambda-value (third code)) =>
	    (lambda (arglist)
	      (fnlike-generator doc lines name arglist)))
	   (else
	    (gen-variable-def doc lines name (cddr code))))))))

(define handle-define (make-define-like-handler gen-function-def))
(define handle-defmacro (make-define-like-handler gen-macro-def))

(define (handle-define-record doc lines code)
  (let ((recname (second code))
	(varnames (cddr code)))
    (doc
     'append-item!
     `(classsynopsis (ooclass (classname ,recname))
		     ,@(map (lambda (varname)
			      `(fieldsynopsis (varname ,varname)))
			    varnames)))
    (gen-body-text doc lines)))

(define (handle-undefined doc lines code)
  (doc
   'append-item!
   `(programlisting ,(->string code)))
  (gen-body-text doc lines))

(define head-handlers
  `((define ,handle-define)
    (defmacro ,handle-defmacro)
    (define-macro ,handle-defmacro)
    (define-record ,handle-define-record)))

;;@ Processes a single <command>docscm</command> comment block.
(define (gen-block cblk doc)
  (let ((cmd (comment-block-command cblk))
	(lines (comment-block-lines cblk))
	(code (comment-block-code cblk)))
    (cond
     (cmd
      (case (first cmd)
	((section)	(doc 'new-section!
			     (second cmd)
			     (and (not (null? (cddr cmd)))
				  (third cmd)))
			(gen-body-text doc lines))
	((variable var)	(gen-variable-def doc lines (second cmd) (cddr cmd)))
	((function fn)	(gen-function-def doc lines
					  (car (second cmd))
					  (cdr (second cmd))))
	((macro)	(gen-macro-def doc lines
				       (car (second cmd))
				       (cdr (second cmd))))
	((sxml)		(gen-sxml doc lines))
	((title)	(doc 'set-title! (second cmd))
			(gen-body-text doc lines))
	(else (error "Unrecognised @-command" cmd))))
     (code
      (cond
       ((assq (first code) head-handlers) =>
	(lambda (cell)
	  ((second cell) doc lines code)))
       (else
	(handle-undefined doc lines code))))
     (else
      (gen-body-text doc lines)))))

;;@ Returns an SXML fragment of documentation for file
;;<parameter>filename</parameter>, generated from the comment-blocks
;;in <parameter>items</parameter>.
(define (gen-doc filename items)
  (let* ((basename (string-append (pathname-strip-directory filename)))
	 (doc
	  (let ((title (string-append "Documentation for " basename))
		(rbody '())
		(rsect #f))

	    (define (set-title! t)
	      (set! title t))

	    (define (finish-section!)
	      (if rsect
		  (set! rbody (cons (reverse rsect) rbody)))
	      (set! rsect #f))

	    (define (new-section! title id)
	      (finish-section!)
	      (set! rsect (reverse `(section ,@(if id `((@ (id ,id))) '())
					     (title ,title)))))

	    (define (append-item! sxmlfrag)
	      (if (not rsect)
		  (set! rbody (cons sxmlfrag rbody))
		  (set! rsect (cons sxmlfrag rsect))))

	    (define (render)
	      (let ((indexid (newsym)))
		(finish-section!)
		(list 'quasiquote
		      `(section (@ (id ,basename))
				(title ,title)
; 				(indexterm (@ (id ,indexid))
; 					   (primary (filename ,basename)))
				,@(if (null? rbody)
				      '((para))
				      (reverse! rbody))))))

	    (lambda (op . args)
	      (apply (cond
		      ((assq op `((set-title! ,set-title!)
				  (finish-section! ,finish-section!)
				  (new-section! ,new-section!)
				  (append-item! ,append-item!)
				  (render ,render)))
		       => second)
		      (else (error "Unknown method on doc" op args)))
		     args)))))

    (for-each (lambda (item)
		(gen-block item doc))
	      items)
    (doc 'render)))

;;@ Extracts documentation from the file passed in as an
;;argument. <parameter>filename</parameter> is the file to parse. The
;;function returns a quasiquoted SXML DocBook expression.
(define extract-doc
  (let ((first-re	"[ \t]*(;+)@([ \t]*)(.*)")
	(next-re	"[ \t]*(;+)(.*)"))

    (define (escape-string str)
      (list->string
       (cons #\"
	     (string-fold-right (lambda (c acc)
				  (case c
				    ((#\" #\\) (cons* #\\ c acc))
				    (else (cons c acc))))
				'(#\")
				str))))

    (define (line-filter line acc)
      (let ((semi-count (car acc))
	    (lines (cdr acc)))
	(if semi-count
	    (let* ((still-in		(string-match next-re line))
		   (new-semi-count	(and still-in (string-length (second still-in))))
		   (text		(and still-in (third still-in))))
	      (cond
	       ((and still-in (= semi-count new-semi-count))
		(cons* semi-count (escape-string text) lines))
	       (else
		(cons* #f (string-append ")\n" line) lines))))
	    (let* ((is-first		(string-match first-re line))
		   (new-semi-count	(and is-first (string-length (second is-first))))
		   (ws			(and is-first (third is-first)))
		   (text		(and is-first (fourth is-first))))
	      (if is-first
		  (let ((command (and (zero? (string-length ws))
				      (not (zero? (string-length text)))
				      (string-append "(" text ")"))))
		    (cons* new-semi-count
			   (if command
			       (string-append "#,(comment-block " command)
			       (string-append "#,(comment-block #f " (escape-string text)))
			   lines))
		  (cons* #f line lines))))))

    (define (filter-lines lines)
      (let* ((acc (fold line-filter (cons #f '()) lines))
	     (semi-count (car acc))
	     (new-lines (cdr acc)))
	(reverse
	 (if semi-count
	     (cons ")\n" new-lines)
	     new-lines))))

    (define (main-parse-loop)
      (let loop ((datum (read)))
	(cond
	 ((eof-object? datum) '())
	 ((comment-block? datum)
	  (let ((succ (read)))
	    (cond
	     ((and (pair? succ)
		   (let ((key (car succ)))
		     (and (symbol? key)
			  (or (eq? key 'defmacro)
			      (eq? key 'define)
			      (string=? (substring (symbol->string (car succ)) 0 7)
					"define-")))))
	      (comment-block-code-set! datum succ)
	      (cons datum (loop (read))))
	     (else
	      (cons datum (loop succ))))))
	 (else
	  (loop (read))))))

    (lambda (filename)
      (with-input-from-file filename
	(lambda ()
	  (let* ((contents (string-join (filter-lines (read-lines)) "\n")))
	    (with-input-from-string contents
	      (lambda ()
		(gen-doc filename (main-parse-loop))))))))))

(for-each (compose pretty-print extract-doc) (command-line-arguments))
